<?php

namespace Drupal\oembed_field\Plugin\Field\FieldType;

use Drupal\Core\Link;
use Drupal\Core\StreamWrapper\StreamWrapperInterface;
use Drupal\Component\Utility\Crypt;
use Drupal\Core\Field\FieldItemBase;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\DataDefinition;
use Drupal\Core\Field\Attribute\FieldType;
use Drupal\Core\Url;

/**
 * Field type that stores oEmbed URLs with cached metadata.
 *
 * This field type stores oEmbed URLs and automatically caches provider
 * information without making external API calls during save operations.
 * It supports provider bucket restrictions and thumbnail management.
 */
#[FieldType(
  id: "oembed_url_field",
  label: new TranslatableMarkup("oEmbed URL"),
  description: new TranslatableMarkup("Stores oEmbed URLs with cached metadata."),
  category: new TranslatableMarkup("general"),
  default_widget: "oembed_url_textfield",
  default_formatter: "oembed_field_default"
)]
class OembedUrlField extends FieldItemBase {

  /**
   * Original field values when loaded from storage.
   *
   * Used to detect changes during preSave operations.
   *
   * @var array
   */
  protected array $original = [];

  /**
   * Prevents usage on media entities.
   */
  public static function isApplicable(FieldStorageDefinitionInterface $field_definition) {
    return $field_definition->getTargetEntityTypeId() !== 'media';
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultFieldSettings() {
    return [
        'file_directory' => \Drupal::moduleHandler()->moduleExists('token') ? 'oembed_thumbnails/[date:custom:Y]-[date:custom:m]' : 'oembed_thumbnails',
        'uri_scheme' => \Drupal::config('system.file')->get('default_scheme') ?? 'public',
        'provider_buckets' => [],
      ] + parent::defaultFieldSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function fieldSettingsForm(array $form, FormStateInterface $form_state) {
    $element = parent::fieldSettingsForm($form, $form_state);

    // Get available provider buckets.
    $provider_buckets = $this->getOembedProviderBuckets();

    if ($provider_buckets) {
      $element['provider_buckets'] = [
        '#type' => 'checkboxes',
        '#title' => $this->t('Allowed provider buckets'),
        '#default_value' => $this->getSetting('provider_buckets') ?: [],
        '#options' => $provider_buckets,
        '#description' => $this->t('Select which types of oEmbed content are allowed. If none are selected, all types are allowed.'),
        '#weight' => 0,
      ];
    }
    else {
      if (!\Drupal::moduleHandler()->moduleExists('oembed_providers')) {
        $url = Url::fromUri('https://www.drupal.org/project/oembed_providers', ['attributes' => ['target' => '_blank']]);
        $warning = $this->t('Install the @link module to control which providers are allowed.', [
          '@link' => Link::fromTextAndUrl('Oembed Providers', $url)->toString(),
        ]);
      }
      else {
        $url = Url::fromRoute('entity.oembed_provider_bucket.collection', ['attributes' => ['target' => '_blank']]);
        $warning = $this->t('You have not set up any @link. All available providers will be allowed.', [
          '@link' => Link::fromTextAndUrl('oembed provider buckets', $url)->toString(),
        ]);
      }
      $element['allowed_buckets_warning'] = [
        '#theme' => 'status_messages',
        '#message_list' => [
          'warning' => [$warning],
        ],
      ];
    }

    $element['thumbnails'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Thumbnail Storage'),
      '#tree' => FALSE,
    ];

    $scheme_options = \Drupal::service('stream_wrapper_manager')->getNames(StreamWrapperInterface::WRITE_VISIBLE);
    $element['thumbnails']['uri_scheme'] = [
      '#type' => 'radios',
      '#title' => $this->t('Storage scheme'),
      '#options' => $scheme_options,
      '#default_value' => $this->getSetting('uri_scheme') ?? \Drupal::config('system.file')->get('default_scheme'),
      '#description' => $this->t('Select where the thumbnail files should be stored. Private file storage has significantly more overhead than public files, but allows restricted access to files within this field.'),
    ];

    $element['thumbnails']['file_directory'] = [
      '#type' => 'textfield',
      '#title' => $this->t('File directory'),
      '#default_value' => $this->getSetting('file_directory'),
      '#description' => $this->t('Subdirectory where thumbnails will be stored. Use tokens like [date:custom:Y]-[date:custom:m] for year/month organization.'),
      '#element_validate' => [[$this, 'validateDirectory']],
      '#required' => TRUE,
    ];

    // Add token support info.
    if (\Drupal::moduleHandler()->moduleExists('token')) {
      $element['thumbnails']['token_help'] = [
        '#theme' => 'token_tree_link',
      ];
    }

    return $element;
  }

  /**
   * Validates the file directory field.
   *
   * Ensures no leading or trailing slashes in the directory path.
   *
   * @param array $element
   *   The form element to validate.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public static function validateDirectory($element, FormStateInterface $form_state) {
    // Ensure no leading/trailing slashes.
    $value = trim($element['#value'], '/');
    $form_state->setValueForElement($element, $value);
  }

  /**
   * {@inheritdoc}
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition) {
    $properties['value'] = DataDefinition::create('string')
      ->setLabel(t('oEmbed URL'))
      ->setDescription(t('The oEmbed URL'));

    $properties['url_hash'] = DataDefinition::create('string')
      ->setLabel(t('URL Hash'))
      ->setDescription(t('Hash of the URL for lookups'));

    $properties['provider'] = DataDefinition::create('string')
      ->setLabel(t('Provider'))
      ->setDescription(t('The oEmbed provider name'));

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public static function schema(FieldStorageDefinitionInterface $field_definition) {
    return [
      'columns' => [
        'value' => [
          'type' => 'text',
          'size' => 'normal',
          'not null' => FALSE,
        ],
        'url_hash' => [
          'type' => 'varchar',
          'length' => 64,
          'not null' => FALSE,
        ],
        'provider' => [
          'type' => 'varchar',
          'length' => 255,
          'not null' => FALSE,
        ],
      ],
      'indexes' => [
        'url_hash' => ['url_hash'],
        'provider' => ['provider'],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function setValue($values, $notify = TRUE) {
    // Store the original values when the field is loaded.
    if (empty($this->original) && is_array($values)) {
      $this->original = $values;
    }

    // Handle string input.
    if (is_string($values)) {
      $values = ['value' => $values];
    }

    parent::setValue($values, $notify);
  }

  /**
   * {@inheritdoc}
   */
  public function preSave() {
    parent::preSave();

    $url = $this->get('value')->getValue();

    // Skip if no URL.
    if (empty($url)) {
      $this->set('url_hash', NULL);
      $this->set('provider', NULL);
      return;
    }

    // Calculate and store hash.
    $hash = Crypt::hashBase64($url);
    $this->set('url_hash', $hash);

    // Get provider from oEmbed provider repository (no API call)
    $provider = $this->getProviderFromRepository($url);
    $this->set('provider', $provider);
  }

  /**
   * Gets provider name from oEmbed provider repository.
   *
   * Checks the URL against provider endpoints to determine which oEmbed
   * provider should handle it. Does not make external API calls.
   *
   * @param string $url
   *   The oEmbed URL to check.
   *
   * @return string|null
   *   The provider name if found, NULL otherwise.
   */
  protected function getProviderFromRepository($url) {
    try {
      $manager = \Drupal::service('oembed_field.manager');
      $provider_repository = \Drupal::service('media.oembed.provider_repository');

      // Get allowed buckets from field settings.
      $allowed_buckets = array_filter($this->getSetting('provider_buckets') ?: []);

      // Get providers based on buckets.
      if (!empty($allowed_buckets)) {
        // Get only providers from allowed buckets.
        $provider_names = $manager->getProviderNamesFromBuckets($allowed_buckets);

        foreach ($provider_names as $provider_name) {
          $provider = $provider_repository->get($provider_name);
          foreach ($provider->getEndpoints() as $endpoint) {
            if ($endpoint->matchUrl($url)) {
              return $provider->getName();
            }
          }
        }
      }
      else {
        // No buckets configured, check all providers.
        $providers = $provider_repository->getAll();

        foreach ($providers as $provider) {
          foreach ($provider->getEndpoints() as $endpoint) {
            if ($endpoint->matchUrl($url)) {
              return $provider->getName();
            }
          }
        }
      }
    }
    catch (\Exception $e) {
      // URL doesn't match any provider.
      \Drupal::logger('oembed_field')->debug('No provider found for URL: @url', [
        '@url' => $url,
      ]);
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function isEmpty() {
    return empty($this->value);
  }

  /**
   * {@inheritdoc}
   */
  public function getConstraints() {
    $constraints = parent::getConstraints();

    $allowed_buckets = $this->getSetting('provider_buckets') ?: [];

    $constraint_manager = \Drupal::typedDataManager()->getValidationConstraintManager();
    $constraints[] = $constraint_manager->create('OembedFieldUrl', [
      'allowed_buckets' => $allowed_buckets,
    ]);

    return $constraints;
  }

  /**
   * Retrieves available oEmbed provider buckets and their labels.
   *
   * Loads provider bucket entities from the oembed_providers module
   * if available.
   *
   * @return array
   *   An associative array of bucket IDs to labels, or empty array if
   *   oembed_providers module is not installed.
   */
  public function getOembedProviderBuckets() {
    $options = [];
    if (\Drupal::moduleHandler()->moduleExists('oembed_providers')) {
      if ($buckets = \Drupal::entityTypeManager()->getStorage('oembed_provider_bucket')->loadMultiple()) {
        foreach ($buckets as $bucket) {
          $options[$bucket->id()] = $bucket->label();
        }
      }
    }
    return $options;
  }

  /**
   * Gets thumbnail storage settings for this field.
   *
   * @return array
   *   An array containing:
   *   - uri_scheme: The file storage scheme (public, private, etc.)
   *   - file_directory: The directory path for thumbnails
   */
  public function getThumbnailSettings() {
    return [
      'uri_scheme' => $this->getSetting('uri_scheme'),
      'file_directory' => $this->getSetting('file_directory'),
    ];
  }

  /**
   * Gets the thumbnail file entity for this field item.
   *
   * @return \Drupal\file\FileInterface|null
   *   The file entity if it exists, NULL otherwise.
   */
  public function getThumbnailFile() {
    return \Drupal::service('oembed_field.thumbnail_manager')->getThumbnailFileForField($this);
  }

}
