<?php

namespace Drupal\oembed_field\Plugin\Field\FieldWidget;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Plugin implementation of the 'oembed_url_textfield' widget.
 *
 * @FieldWidget(
 *   id = "oembed_url_textfield",
 *   label = @Translation("oEmbed URL textfield"),
 *   field_types = {
 *     "oembed_url_field"
 *   }
 * )
 */
class OembedUrlTextfield extends WidgetBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'show_preview' => FALSE,
      'image_style' => '',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = parent::settingsForm($form, $form_state);

    $elements['show_preview'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show preview'),
      '#default_value' => $this->getSetting('show_preview'),
      '#description' => $this->t('Show a preview of the oEmbed content when editing.'),
    ];

    $image_styles = image_style_options(FALSE);
    $elements['image_style'] = [
      '#title' => $this->t('Image style'),
      '#type' => 'select',
      '#default_value' => $this->getSetting('image_style'),
      '#empty_option' => $this->t('None (original image)'),
      '#options' => $image_styles,
      '#description' => $this->t('Select an image style to apply to the thumbnail.'),
      '#states' => [
        'visible' => [
          ':input[name*="[show_preview]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];

    if ($this->getSetting('show_preview')) {
      $summary[] = $this->t('Preview enabled: @style', [
        '@style' => $this->getSetting('image_style') ?: 'original',
      ]);
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $item = $items[$delta];

    // Build the main element container.
    $element = [
      '#type' => 'container',
      '#attributes' => ['class' => ['oembed-url-field-widget']],
    ] + $element;

    // Add a default description indicating the supported providers in the field.
    $allowed_buckets = array_values(array_filter($this->getFieldSetting('provider_buckets') ?: []));

    if (!empty($allowed_buckets) && \Drupal::service('oembed_field.manager')) {
      $providers = \Drupal::service('oembed_field.manager')->getProviderNamesFromBuckets($allowed_buckets);
      $description = t('Supported providers: @providers', [
        '@providers' => implode(', ', $providers),
      ]);
    }
    else {
      $description = '';
    }

    // If we have a description on the element, display a list of descriptions.
    if ($element['#description'] && $description) {
      $description = [
        '#theme' => 'item_list',
        '#items' => [
          $description,
          $element['#description'],
        ],
      ];
    }

    $element['value'] = [
      '#type' => 'textfield',
      '#title' => $element['#title'],
      '#title_display' => $element['#title_display'],
      '#description' => $description ?: $element['#description'],
      '#default_value' => $item->value ?? '',
      '#required' => $element['#required'],
      '#attributes' => [
        'class' => ['oembed-field-input'],
      ],
    ];

    // Add preview if enabled and we have data.
    if ($this->getSetting('show_preview') && !empty($item->value)) {
      $element['preview'] = $this->buildPreview($item);
    }

    return $element;
  }

  /**
   * Build preview render array.
   */
  protected function buildPreview($item) {
    $preview = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['oembed-field-preview'],
      ],
    ];

    // Show preview content if we have metadata.
    if (!empty($item->title) || !empty($item->provider)) {
      $preview['content'] = [
        '#type' => 'container',
        '#attributes' => [
          'class' => ['oembed-preview-content'],
        ],
      ];

      // Get thumbnail file from centralized storage.
      $thumbnail_file = $item->getThumbnailFile();

      if ($thumbnail_file) {
        if ($this->getSetting('image_style')) {
          $preview['content']['thumbnail'] = [
            '#theme' => 'image_style',
            '#style_name' => $this->getSetting('image_style'),
            '#uri' => $thumbnail_file->getFileUri(),
            '#alt' => $item->title ?: $this->t('Preview'),
          ];
        }
        else {
          $preview['content']['thumbnail'] = [
            '#theme' => 'image',
            '#uri' => $thumbnail_file->getFileUri(),
            '#alt' => $item->title ?: $this->t('Preview'),
            '#attributes' => [
              'class' => ['oembed-preview-thumbnail'],
              'style' => 'max-width: 200px;',
            ],
          ];
        }
      }
      elseif (!empty($item->thumbnail_url)) {
        // Show remote thumbnail if local not available yet.
        $preview['content']['thumbnail'] = [
          '#markup' => '<img src="' . $item->thumbnail_url . '" style="max-width: 200px;" alt="' . ($item->title ?: $this->t('Preview')) . '" class="oembed-preview-thumbnail" />',
        ];
      }

      // Show metadata.
      $metadata = [];

      if (!empty($item->title)) {
        $metadata[] = '<strong>' . $item->title . '</strong>';
      }

      if (!empty($item->provider)) {
        $metadata[] = $this->t('Provider: @provider', ['@provider' => $item->provider]);
      }

      if (!empty($metadata)) {
        $preview['content']['metadata'] = [
          '#type' => 'html_tag',
          '#tag' => 'div',
          '#attributes' => ['class' => ['oembed-preview-metadata']],
          '#value' => implode('<br>', $metadata),
        ];
      }
    }

    return $preview;
  }

  /**
   * {@inheritdoc}
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state) {
    $items = $this->fieldDefinition->getName();
    $entity = $form_state->getFormObject()->getEntity();

    foreach ($values as $delta => &$value) {
      $url = '';
      if (is_array($value) && isset($value['value'])) {
        $url = trim($value['value']);
      }
      elseif (is_string($value)) {
        $url = trim($value);
      }

      if ($url !== '') {
        // Preserve existing field data if available.
        $existing_data = [];
        if ($entity && $entity->hasField($items) && isset($entity->get($items)[$delta])) {
          $existing_item = $entity->get($items)[$delta];
          $existing_data = [
            'url_hash' => $existing_item->url_hash,
            'provider' => $existing_item->provider,
          ];
        }

        // Merge with new URL.
        $values[$delta] = array_merge($existing_data, ['value' => $url]);
      }
      else {
        // Remove empty values.
        unset($values[$delta]);
      }
    }

    // Re-index array.
    return array_values($values);
  }

}
