<?php

namespace Drupal\oembed_field\Plugin\Validation\Constraint;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\media\OEmbed\ProviderRepositoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\ConstraintValidator;

/**
 * Validates the OembedFieldBucketConstraint.
 */
class OembedFieldUrlConstraintValidator extends ConstraintValidator implements ContainerInjectionInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The oEmbed provider repository.
   *
   * @var \Drupal\media\OEmbed\ProviderRepositoryInterface
   */
  protected $providerRepository;

  /**
   * Constructs the validator.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   * @param \Drupal\media\OEmbed\ProviderRepositoryInterface $provider_repository
   *   The oEmbed provider repository service.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, ModuleHandlerInterface $module_handler, ProviderRepositoryInterface $provider_repository) {
    $this->entityTypeManager = $entity_type_manager;
    $this->moduleHandler = $module_handler;
    $this->providerRepository = $provider_repository;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('module_handler'),
      $container->get('media.oembed.provider_repository')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function validate($value, Constraint $constraint) {
    // Get the URL from the field item.
    $url = NULL;
    if (is_object($value) && method_exists($value, 'getValue')) {
      $item_value = $value->getValue();
      $url = $item_value['value'] ?? NULL;
    }
    elseif (is_string($value)) {
      $url = $value;
    }

    if (empty($url)) {
      // Let required validation handle empty values.
      return;
    }

    // Check if URL has changed - if not, skip validation.
    if (isset($value->original)) {
      $original_url = $value->original->get('value')->getValue();
      // If URL hasn't changed, skip validation.
      if ($url === $original_url) {
        return;
      }
    }

    // Get allowed buckets from constraint.
    $allowed_buckets = array_filter($constraint->allowed_buckets ?: []);

    // Check if URL is valid oEmbed.
    if (!$this->isValidOembedUrl($url, $allowed_buckets)) {
      $message = empty($allowed_buckets)
        ? 'The URL @url is not a valid oEmbed URL.'
        : $constraint->message;

      $this->context->addViolation($message, [
        '@url' => $url,
      ]);
    }
  }

  /**
   * Check if URL is a valid oEmbed URL.
   */
  protected function isValidOembedUrl(string $url, array $allowed_buckets): bool {
    // If no buckets configured, check against ALL providers.
    if (empty($allowed_buckets)) {
      $providers = $this->providerRepository->getAll();
    }
    else {
      // Get only providers from allowed buckets.
      $providers = $this->getProvidersFromBuckets($allowed_buckets);
    }

    // Check if URL matches any provider.
    foreach ($providers as $provider_info) {
      foreach ($provider_info->getEndpoints() as $endpoint) {
        if ($endpoint->matchUrl($url)) {
          return TRUE;
        }
      }
    }

    return FALSE;
  }

  /**
   * Returns an array of all providers in $buckets.
   */
  private function getProvidersFromBuckets(array $buckets): array {
    $enabled_providers = [];
    $buckets = $this->entityTypeManager->getStorage('oembed_provider_bucket')
      ->loadMultiple($buckets);
    foreach ($buckets as $bucket) {
      $enabled_providers = array_merge($enabled_providers, $bucket->get('providers'));
    }
    $enabled_providers = array_unique($enabled_providers);
    foreach ($enabled_providers as $provider) {
      $providers[] = $this->providerRepository->get($provider);
    }
    return $providers;
  }

}
