<?php

declare(strict_types=1);

namespace Drupal\oembed_tweak\OEmbed;

use Drupal\Component\Plugin\Exception\PluginException;
use Drupal\Component\Plugin\PluginManagerInterface;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\media\OEmbed\ProviderRepositoryInterface;
use Drupal\media\OEmbed\ResourceException;
use Drupal\media\OEmbed\ResourceFetcher as BaseResourceFetcher;
use Drupal\oembed_tweak\OEmbedTweak\TweakPluginManager;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\RequestOptions;
use Psr\Http\Client\ClientExceptionInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Provides an overridden OEmbed resource fetcher.
 *
 * This overrides to the core class to apply tweaks to the fetched resource
 * data.
 */
class ResourceFetcher extends BaseResourceFetcher {

  /**
   * A static cache of tweak instances.
   *
   * @var ?\Drupal\oembed_tweak\OEmbedTweak\TweakInterface[]
   */
  protected ?array $tweaks = NULL;

  public function __construct(
    ClientInterface $httpClient,
    ProviderRepositoryInterface $providers,
    #[Autowire(service: 'cache.default')]
    CacheBackendInterface $cacheBackend,
    #[Autowire(service: TweakPluginManager::class)]
    protected PluginManagerInterface $tweakManager,
    int $timeout = 5,
  ) {
    parent::__construct($httpClient, $providers, $cacheBackend, $timeout);
  }

  /**
   * {@inheritdoc}
   */
  public function fetchResource($url) {
    $cache_id = "media:oembed_resource:$url";

    $cached = $this->cacheBackend->get($cache_id);
    if ($cached) {
      return $this->createResource($cached->data, $url);
    }

    try {
      $response = $this->httpClient->request('GET', $url, [
        RequestOptions::TIMEOUT => $this->timeout,
      ]);
    }
    catch (ClientExceptionInterface $e) {
      throw new ResourceException('Could not retrieve the oEmbed resource.', $url, [], $e);
    }

    [$format] = $response->getHeader('Content-Type');
    $content = (string) $response->getBody();

    if (strstr($format, 'text/xml') || strstr($format, 'application/xml')) {
      $data = $this->parseResourceXml($content, $url);
    }
    // By default, try to parse the resource data as JSON.
    else {
      $data = Json::decode($content);

      if (json_last_error() !== JSON_ERROR_NONE) {
        throw new ResourceException('Error decoding oEmbed resource: ' . json_last_error_msg(), $url);
      }
    }
    if (empty($data) || !is_array($data)) {
      throw new ResourceException('The oEmbed resource could not be decoded.', $url);
    }

    foreach ($this->getTweaks() as $tweak) {
      if ($tweak->isApplicable($data, $url)) {
        $data = $tweak->tweak($data, $url);
      }
    }

    $this->cacheBackend->set($cache_id, $data);

    return $this->createResource($data, $url);
  }

  /**
   * Gets all available OEmbed tweaks.
   *
   * @return \Drupal\oembed_tweak\OEmbedTweak\TweakInterface[]
   *   A list of all OEmbed tweaks.
   */
  protected function getTweaks(): array {
    if ($this->tweaks === NULL) {
      $this->tweaks = [];

      foreach (array_keys($this->tweakManager->getDefinitions()) as $tweakId) {
        try {
          $this->tweaks[$tweakId] = $this->tweakManager->createInstance($tweakId);
        }
        catch (PluginException) {
        }
      }
    }
    return $this->tweaks;
  }

}
