<?php

declare(strict_types=1);

namespace Drupal\Tests\oembed_tweak\Kernel;

use Drupal\Core\Extension\ModuleInstallerInterface;
use Drupal\Core\Template\TwigEnvironment;
use Drupal\Core\Url;
use Drupal\KernelTests\KernelTestBase;
use Drupal\media\OEmbed\Provider;
use Drupal\media\OEmbed\ProviderException;
use Drupal\media\OEmbed\ProviderRepositoryInterface;
use Drupal\media\OEmbed\Resource;
use Drupal\media\OEmbed\ResourceException;
use Drupal\media\OEmbed\ResourceFetcherInterface;
use Drupal\media\OEmbed\UrlResolverInterface;
use Drupal\oembed_tweak\OEmbedTweak\TweakInterface;
use Drupal\oembed_tweak\OEmbedTweak\TweakPluginManager;
use Drupal\Tests\oembed_tweak\TestTools\DiffLineType;
use Drupal\Tests\oembed_tweak\TestTools\Fixtures;
use Drupal\Tests\oembed_tweak\TestTools\FixturesHttpClient;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;
use Jfcherng\Diff\DiffHelper;
use PHPUnit\Framework\Attributes\RunTestsInSeparateProcesses;

/**
 * Tests the necessity for and the functionality of the OEmbed Tweaks.
 */
#[RunTestsInSeparateProcesses]
class OEmbedTweakProviderTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['media', 'user'];

  /**
   * The test fixtures helper.
   *
   * @var \Drupal\Tests\oembed_tweak\TestTools\Fixtures
   */
  protected Fixtures $fixtures;

  /**
   * The HTTP client to access the test fixtures.
   *
   * @var \Drupal\Tests\oembed_tweak\TestTools\FixturesHttpClient
   */
  protected FixturesHttpClient $httpClient;

  /**
   * The OEmbed provider repository.
   *
   * @var \Drupal\media\OEmbed\ProviderRepositoryInterface
   */
  protected ProviderRepositoryInterface $providerRepository;

  /**
   * The OEmbed URL resolver.
   *
   * @var \Drupal\media\OEmbed\UrlResolverInterface
   */
  protected UrlResolverInterface $urlResolver;

  /**
   * The OEmbed resource fetcher.
   *
   * @var \Drupal\media\OEmbed\ResourceFetcherInterface
   */
  protected ResourceFetcherInterface $resourceFetcher;

  /**
   * The module installer.
   *
   * @var \Drupal\Core\Extension\ModuleInstallerInterface
   */
  protected ModuleInstallerInterface $moduleInstaller;

  /**
   * The number of untested providers.
   *
   * @var int
   */
  protected int $untestedProviderCount = 358;

  /**
   * The minimum thumbnail width for a response to be considered successful.
   *
   * @var int
   */
  protected int $minimumThumbnailWidth = 720;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installConfig('media');

    $this->fixtures = new Fixtures();

    $this->container->set('http_client', new FixturesHttpClient());
    $this->httpClient = $this->container->get(ClientInterface::class);

    $this->providerRepository = $this->container->get(ProviderRepositoryInterface::class);
    $this->urlResolver        = $this->container->get(UrlResolverInterface::class);
    $this->resourceFetcher    = $this->container->get(ResourceFetcherInterface::class);
    $this->moduleInstaller    = $this->container->get(ModuleInstallerInterface::class);
  }

  /**
   * Tests that providers require tweaks and that the tweaks work as expected.
   */
  public function testProviders(): void {
    try {
      $allProviders = $this->providerRepository->getAll();
    }
    catch (ProviderException $exception) {
      $this->fail($exception->getMessage());
    }

    $testUrlInfo = $this->fixtures->getTestUrlInfo();
    $untestedProviders = array_diff_key(
      $allProviders,
      $testUrlInfo,
    );
    $this->assertCount($this->untestedProviderCount, $untestedProviders);

    $testResourceUrls = [];
    foreach ($testUrlInfo as $providerName => $providerTestUrls) {
      $this->assertArrayHasKey($providerName, $allProviders);
      $provider = $allProviders[$providerName];

      $this->assertInstanceOf(Provider::class, $provider);
      $schemes = [];
      foreach ($provider->getEndpoints() as $endpoint) {
        $schemes = [...$schemes, ...$endpoint->getSchemes()];
      }

      foreach ($providerTestUrls['urls'] as $testUrl => $expectedResult) {
        $schemeMatch = FALSE;
        foreach ($schemes as $schemeIndex => $scheme) {
          /* @see \Drupal\media\OEmbed\Endpoint::matchUrl() */
          // Improve the regular expression to disallow slashes in wildcards so
          // that wildcards only match a single path component.
          // @todo Find out what the specification says about this and
          //   potentially fix this in core.
          $regexp = str_replace(['.', '*', '?'], ['\.', '[^/]*', '\?'], $scheme);
          if (preg_match("|^$regexp$|", $testUrl)) {
            $schemeMatch = TRUE;
            unset($schemes[$schemeIndex]);
            break;
          }
        }
        if (!$schemeMatch) {
          $this->fail("$testUrl does not match any $providerName scheme");
        }

        $testResourceUrls[$testUrl] = $this->urlResolver->getResourceUrl($testUrl);

        try {
          $resource = $this->resourceFetcher->fetchResource($testResourceUrls[$testUrl]);
          if ($expectedResult['status'] === 'resource_exception') {
            $this->fail("Valid resource returned where a resource exception was expected for test URL $testUrl");
          }
        }
        catch (ResourceException $exception) {
          if ($expectedResult['status'] === 'resource_exception') {
            $this->assertEquals($expectedResult['exception_message'], $exception->getMessage());
            continue;
          }
          else {
            $this->fail("Resource exception raised where a valid resource was expected for test URL $testUrl: {$exception->getMessage()}");
          }
        }

        $this->assertInstanceOf(Resource::class, $resource);

        if ($resource->getType() === Resource::TYPE_VIDEO) {
          $thumbnailUrl = $resource->getThumbnailUrl();
          $this->assertInstanceOf(Url::class, $thumbnailUrl, "Missing thumbnail URL for resource of type '{$resource->getType()}' with URL $testResourceUrls[$testUrl]");

          if ($expectedResult['status'] === 'small_thumbnail') {
            $this->assertEquals($expectedResult['thumbnail_width'], $resource->getThumbnailWidth(), "Incorrect thumbnail width for resource with URL $testResourceUrls[$testUrl]");
          }
          else {
            try {
              $thumbnailResponse = $this->httpClient->request('GET', $thumbnailUrl->toString());
            }
            catch (RequestException $exception) {
              $this->fail("Error fetching thumbnail URL {$thumbnailUrl->toString()}: {$exception->getMessage()}");
            }

            $this->assertEquals(200, $thumbnailResponse->getStatusCode(), "Incorrect status code for response to thumbnail URL {$thumbnailUrl->toString()}");
            $this->assertGreaterThanOrEqual($this->minimumThumbnailWidth, $resource->getThumbnailWidth(), "Thumbnail width for resource with URL $testResourceUrls[$testUrl] less than {$this->minimumThumbnailWidth}");

            [$width] = getimagesizefromstring((string) $thumbnailResponse);
            $this->assertEquals($resource->getThumbnailWidth(), $width, "The actual thumbnail width for resource with URL $testResourceUrls[$testUrl] does not match the reported width");
          }
        }

        $authorUrl = $resource->getAuthorUrl();
        if ($authorUrl) {
          try {
            $authorResponse = $this->httpClient->request('HEAD', $authorUrl->toString());
          }
          catch (RequestException $exception) {
            $this->fail("Error fetching author URL {$authorUrl->toString()}: {$exception->getMessage()}");
          }

          $this->assertEquals(200, $authorResponse->getStatusCode(), "Incorrect status code for response to author URL {$authorUrl->toString()}");
        }
      }

      $this->assertEquals($providerTestUrls['missing_schemes'] ?? [], array_values($schemes));
    }

    $this->moduleInstaller->install(['oembed_tweak']);
    // @todo Remove this when
    //   https://www.drupal.org/project/drupal/issues/3042423 is fixed.
    $this->container->set('http_client', new FixturesHttpClient());
    $this->httpClient = $this->container->get(ClientInterface::class);
    $this->resourceFetcher = $this->container->get(ResourceFetcherInterface::class);

    foreach ($testUrlInfo as $providerName => $providerTestUrls) {
      foreach ($providerTestUrls['urls'] as $testUrl => $expectedResult) {
        try {
          $resource = $this->resourceFetcher->fetchResource($testResourceUrls[$testUrl]);
        }
        catch (ResourceException $exception) {
          $previousMessage = $exception->getPrevious() ? " (Previous exception: {$exception->getPrevious()->getMessage()})" : '';
          $this->fail("Resource exception for provider $providerName with test URL $testUrl despite tweaks: {$exception->getMessage()}$previousMessage");
        }

        $this->assertInstanceOf(Resource::class, $resource);

        if ($resource->getType() === Resource::TYPE_VIDEO) {
          $thumbnailUrl = $resource->getThumbnailUrl();
          $this->assertInstanceOf(Url::class, $thumbnailUrl, "Missing thumbnail URL for resource of type '{$resource->getType()}' with URL $testUrl despite tweaks");

          try {
            $thumbnailResponse = $this->httpClient->request('GET', $thumbnailUrl->toString());
          }
          catch (RequestException $exception) {
            $this->fail("Error fetching thumbnail URL {$thumbnailUrl->toString()} despite tweaks: {$exception->getMessage()}");
          }

          $this->assertEquals(200, $thumbnailResponse->getStatusCode(), "Incorrect status code for response to thumbnail URL {$thumbnailUrl->toString()} despite tweaks");
          $this->assertGreaterThanOrEqual($this->minimumThumbnailWidth, $resource->getThumbnailWidth(), "Thumbnail width for resource with URL $testUrl less than {$this->minimumThumbnailWidth} despite tweaks");

          [$width] = getimagesizefromstring((string) $thumbnailResponse->getBody());
          $this->assertEquals($resource->getThumbnailWidth(), $width, "The actual thumbnail width for resource with URL $testUrl does not match the reported width despite tweaks");
        }

        $authorUrl = $resource->getAuthorUrl();
        if ($authorUrl) {
          try {
            $authorResponse = $this->httpClient->request('HEAD', $authorUrl->toString());
          }
          catch (RequestException $exception) {
            $this->fail("Error fetching author URL $testUrl despite tweaks: {$exception->getMessage()}");
          }

          $this->assertEquals(200, $authorResponse->getStatusCode(), "Incorrect status code for response to author URL {$authorUrl->toString()} despite tweaks");
        }
      }
    }

    if (!empty(getenv('OEMBED_TWEAK_GENERATE_REPORT'))) {
      $tweakManager = $this->container->get(TweakPluginManager::class);
      $providerTweaks = [];
      $allTweaks = [];
      foreach ($tweakManager->getDefinitions() as $tweakId => $tweakDefinition) {
        foreach ($tweakDefinition['providers'] as $providerName) {
          $providerTweaks += [$providerName => []];
          $providerTweaks[$providerName][$tweakId] = $tweakDefinition['label'];
        }
        $allTweaks[$tweakId] = $tweakManager->createInstance($tweakId);
      }

      $workingProviderNames = array_keys(array_filter($testUrlInfo, function ($providerTestUrls) {
        foreach ($providerTestUrls['urls'] as $expectedResult) {
          if ($expectedResult['status'] !== 'success') {
            return FALSE;
          }
        }
        return TRUE;
      }));
      $workingProviders = array_intersect_key($allProviders, array_flip($workingProviderNames));

      $tweakedProviderNames = array_diff(array_keys($testUrlInfo), $workingProviderNames);
      $tweakedProvidersInfo = [];
      foreach ($tweakedProviderNames as $tweakedProviderName) {
        $tweakedProvidersInfo[$tweakedProviderName] = [
          'provider' => $allProviders[$tweakedProviderName],
          'tweaks' => $providerTweaks[$tweakedProviderName],
          'examples' => [],
        ];
        foreach (array_keys($testUrlInfo[$tweakedProviderName]['urls']) as $testUrl) {
          $tweakedProvidersInfo[$tweakedProviderName]['examples'][$testUrl] = [
            'resource_url' => $testResourceUrls[$testUrl],
            'diff' => [],
          ];

          $body = (string) $this->httpClient->request('GET', $testResourceUrls[$testUrl])->getBody();
          $before = json_decode($body, TRUE);
          $after = $before;
          foreach (array_keys($providerTweaks[$tweakedProviderName]) as $tweakId) {
            $tweak = $allTweaks[$tweakId];
            assert($tweak instanceof TweakInterface);
            $after = $tweak->tweak($after, $testUrl);
          }

          $keys = array_unique([...array_keys($before), ...array_keys($after)]);
          foreach ($keys as $key) {
            assert(is_string($key));
            if (!isset($before[$key])) {
              assert(isset($after[$key]));
              $diff = [
                'type'   => DiffLineType::Addition->value,
                'before' => '',
                'after'  => $after[$key],
              ];
            }
            elseif (!isset($after[$key])) {
              $diff = [
                'type'   => DiffLineType::Deletion->value,
                'before' => $before[$key],
                'after'  => '',
              ];
            }
            elseif ($before[$key] === $after[$key]) {
              $diff = [
                'type' => DiffLineType::Context->value,
                'before' => $before[$key],
                'after' => $after[$key],
              ];
            }
            else {
              $stringDiff = json_decode(DiffHelper::calculate(
                old: $before[$key],
                new: $after[$key],
                renderer: 'JsonHtml',
                rendererOptions: [
                  'detailLevel' => 'char',
                ],
              ));
              assert(is_array($stringDiff) && (count($stringDiff) === 1));
              assert(isset($stringDiff[0]) && is_array($stringDiff[0]) && (count($stringDiff[0]) === 1));
              assert(isset($stringDiff[0][0]) && is_object($stringDiff[0][0]));
              assert(isset($stringDiff[0][0]->old) && is_object($stringDiff[0][0]->old));
              assert(isset($stringDiff[0][0]->old->lines) && is_array($stringDiff[0][0]->old->lines) && (count($stringDiff[0][0]->old->lines) === 1));
              assert(isset($stringDiff[0][0]->old->lines) && is_array($stringDiff[0][0]->old->lines) && (count($stringDiff[0][0]->old->lines) === 1));
              assert(isset($stringDiff[0][0]->old->lines[0]));
              assert(isset($stringDiff[0][0]->new) && is_object($stringDiff[0][0]->new));
              assert(isset($stringDiff[0][0]->new->lines) && is_array($stringDiff[0][0]->new->lines) && (count($stringDiff[0][0]->new->lines) === 1));
              assert(isset($stringDiff[0][0]->new->lines) && is_array($stringDiff[0][0]->new->lines) && (count($stringDiff[0][0]->new->lines) === 1));
              assert(isset($stringDiff[0][0]->new->lines[0]));
              $diff = [
                'type' => DiffLineType::Change->value,
                'before' => $stringDiff[0][0]->old->lines[0],
                'after' => $stringDiff[0][0]->new->lines[0],
              ];
            }
            $tweakedProvidersInfo[$tweakedProviderName]['examples'][$testUrl]['diff'][$key] = $diff;
          }
        }
      }

      $twigEnvironment = $this->container->get(TwigEnvironment::class);
      $this->fixtures->generateProviderReport($twigEnvironment, $tweakedProvidersInfo, $workingProviders, $untestedProviders);
    }
  }

}
