<?php

declare(strict_types=1);

namespace Drupal\Tests\oembed_tweak\TestTools;

use Drupal\Component\Serialization\Yaml;
use Drupal\Core\Template\TwigEnvironment;
use Psr\Http\Message\UriInterface;

/**
 * Provides a helper for the test fixtures.
 */
class Fixtures {

  /**
   * The directory containing the test fixtures.
   *
   * This is the "tests/fixtures" directory withing the OEmbed Tweaks module.
   *
   * @var string
   */
  protected string $fixturesDirectory;

  public function __construct() {
    $this->fixturesDirectory = dirname(__DIR__, 2) . '/fixtures';
  }

  /**
   * Gets decoded content of the "test-urls.yml" file.
   *
   * @return array<string, array{urls: array<string, array{status: 'success|small_thumbnail|resource_exception', thumbnail_width?: int, exception_message?: string}>, query_parameters?: array<string, string>, replacements?: array<string, string>, missing_schemes?: string[]}>
   *   The test URL information.
   */
  public function getTestUrlInfo(): array {
    $urlsFile = "$this->fixturesDirectory/test-urls.yml";
    return Yaml::decode(file_get_contents($urlsFile));
  }

  /**
   * Generates a provider report.
   *
   * @param \Drupal\Core\Template\TwigEnvironment $twigEnvironment
   *   The twig environment.
   * @param array<string, array{provider: \Drupal\media\OEmbed\Provider, tweaks: \Drupal\Core\StringTranslation\TranslatableMarkup[]}> $tweakedProvidersInfo
   *   An array of tweaked providers keyed by provider name. Each array contains
   *   the following keys:
   *   - provider: The provider.
   *   - tweaks: A list of tweak labels.
   * @param \Drupal\media\OEmbed\Provider[] $workingProviders
   *   A list of providers that are working without any tweaks.
   * @param \Drupal\media\OEmbed\Provider[] $untestedProviders
   *   A list of providers that are not tested.
   */
  public function generateProviderReport(TwigEnvironment $twigEnvironment, array $tweakedProvidersInfo, array $workingProviders, array $untestedProviders): void {
    $filename = 'report.html';
    $reportFile = "$this->fixturesDirectory/$filename.twig";

    file_get_contents($reportFile);

    file_put_contents($filename, $twigEnvironment->renderInline(file_get_contents($reportFile), [
      'tweaked_providers_info' => $tweakedProvidersInfo,
      'working_providers' => $workingProviders,
      'untested_providers' => $untestedProviders,
      // @todo Add output about per-provider scheme coverage.
    ]));

  }

  /**
   * Gets the file path for a given fixture URI.
   *
   * @param \Psr\Http\Message\UriInterface $uri
   *   The fixture URI.
   *
   * @return string
   *   The file path.
   */
  public function getFixturePath(UriInterface $uri): string {
    $host = str_replace('.', '-', $uri->getHost());
    $path = str_replace('/', '-', ltrim($uri->getPath(), '/'));
    if ($uri->getQuery() !== '') {
      $path .= '--' . str_replace(['.', '/'], '-', str_replace(['=', '%3A//'], '--', $uri->getQuery()));
    }
    return "$this->fixturesDirectory/{$uri->getScheme()}--$host/$path";
  }

  /**
   * Updates a fixture file.
   *
   * @param \Psr\Http\Message\UriInterface $uri
   *   The fixture URI.
   * @param string $body
   *   The response body.
   */
  public function updateFixture(UriInterface $uri, string $body): void {
    $fixturePath = $this->getFixturePath($uri);
    $fixtureDirectory = dirname($fixturePath);
    if (!is_dir($fixtureDirectory)) {
      mkdir($fixtureDirectory);
    }
    file_put_contents($fixturePath, $body);
  }

}
