<?php

declare(strict_types=1);

namespace Drupal\Tests\oembed_tweak\TestTools;

use GuzzleHttp\ClientInterface;
use GuzzleHttp\Promise\Promise;
use GuzzleHttp\Promise\PromiseInterface;
use GuzzleHttp\Psr7\Response;
use GuzzleHttp\Psr7\Uri;
use Psr\Http\Message\RequestInterface;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\UriInterface;

/**
 * Provides a stub HTTP client for fetching the test fixtures.
 */
class FixturesHttpClient implements ClientInterface {

  /**
   * The test fixtures helper.
   *
   * @var \Drupal\Tests\oembed_tweak\TestTools\Fixtures
   */
  protected Fixtures $fixtures;

  public function __construct() {
    $this->fixtures = new Fixtures();
  }

  /**
   * {@inheritdoc}
   */
  public function send(RequestInterface $request, array $options = []): ResponseInterface {
    return $this->respondToUri($request->getMethod(), $request->getUri());
  }

  /**
   * {@inheritdoc}
   */
  public function sendAsync(RequestInterface $request, array $options = []): PromiseInterface {
    return new Promise($this->send(...));
  }

  /**
   * {@inheritdoc}
   */
  public function request(string $method, $uri, array $options = []): ResponseInterface {
    if (is_string($uri)) {
      $uri = new Uri($uri);
    }
    return $this->respondToUri($method, $uri);
  }

  /**
   * {@inheritdoc}
   */
  public function requestAsync(string $method, $uri, array $options = []): PromiseInterface {
    return new Promise($this->request(...));
  }

  /**
   * {@inheritdoc}
   */
  public function getConfig(?string $option = NULL): mixed {
    return ($option === NULL) ? [] : NULL;
  }

  /**
   * Respond to a given URI.
   *
   * @param string $method
   *   The HTTP method of the request.
   * @param \Psr\Http\Message\UriInterface $uri
   *   The URI to respond to.
   *
   * @return \GuzzleHttp\Psr7\Response
   *   The response.
   */
  protected function respondToUri(string $method, UriInterface $uri): Response {
    if (!in_array($method, ['GET', 'HEAD'], TRUE)) {
      return new Response(405, ['Accept' => 'GET']);
    }

    $fixturePath = $this->fixtures->getFixturePath($uri);
    if (!file_exists($fixturePath)) {
      return new Response(404);
    }

    $headers = [];
    if ($method === 'HEAD') {
      $body = '';
    }
    else {
      assert($method === 'GET');
      $body = file_get_contents($fixturePath);
      if (in_array($body[0], ['{', '['], TRUE)) {
        $headers['Content-Type'] = 'application/json';
      }
    }

    return new Response(200, $headers, $body);
  }

}
