<?php

declare(strict_types=1);

namespace Drupal\Tests\oembed_tweak\Unit;

use Drupal\Component\Datetime\Time;
use Drupal\Component\Plugin\Attribute\AttributeInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Http\ClientFactory;
use Drupal\Core\KeyValueStore\KeyValueFactoryInterface;
use Drupal\Core\KeyValueStore\KeyValueStoreInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\media\OEmbed\ProviderRepository;
use Drupal\media\OEmbed\ResourceFetcherInterface;
use Drupal\media\OEmbed\UrlResolver;
use Drupal\oembed_tweak\Attribute\OEmbedTweak;
use Drupal\oembed_tweak\Plugin\OEmbedTweak\AddVersion;
use Drupal\Tests\oembed_tweak\TestTools\Fixtures;
use Drupal\Tests\oembed_tweak\TestTools\FixturesHttpClient;
use Drupal\Tests\UnitTestCase;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Psr7\Uri;
use PHPUnit\Framework\Attributes\DataProvider;

/**
 * Tests that the test fixtures are up-to-date.
 *
 * Because this test involves fetching remote resources, it is skipped unless
 * the "OEMBED_TWEAK_FETCH_FIXTURES" environment variable is set.
 *
 * If you set the "OEMBED_TWEAK_UPDATE_FIXTURES" environment variable, running
 * the test will update the test fixtures.
 */
class OEmbedTweakFixturesTest extends UnitTestCase {

  /**
   * The test fixtures helper.
   *
   * @var \Drupal\Tests\oembed_tweak\TestTools\Fixtures
   */
  protected Fixtures $fixtures;

  /**
   * The HTTP client for remote requests.
   *
   * @var \GuzzleHttp\Client
   */
  protected Client $remoteClient;

  /**
   * The HTTP client to access the test fixtures.
   *
   * @var \Drupal\Tests\oembed_tweak\TestTools\FixturesHttpClient
   */
  protected FixturesHttpClient $fixturesClient;

  /**
   * The remote provider URL.
   *
   * See the "media.settings" configuration object.
   *
   * @var string
   */
  protected string $providersUrl = 'https://oembed.com/providers.json';

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    if (empty(getenv('OEMBED_TWEAK_FETCH_FIXTURES'))) {
      $this->markTestSkipped('Set the OEMBED_TWEAK_FETCH_FIXTURES environment variable to run the fixtures test.');
    }

    parent::setUp();

    $this->fixtures = new Fixtures();

    $this->remoteClient = (new ClientFactory(HandlerStack::create()))->fromOptions();
    $this->fixturesClient = new FixturesHttpClient();

  }

  /**
   * Tests that the providers list fixture is up-to-date.
   */
  public function testProvidersUrl(): void {
    $this->handleFixture('GET', $this->providersUrl);
  }

  /**
   * Provides test data for the providers test.
   *
   * @return array<string, array{providerName: string, testUrlInfo: array{urls: array<string, array{status: 'success|small_thumbnail|resource_exception', thumbnail_width?: int, exception_message?: string}>, query_parameters?: array<string, string>, replacements?: array<string, string>, missing_schemes?: string[]}}>
   *   The test data for the providers test.
   */
  public static function providersProvider(): iterable {
    $fixtures = new Fixtures();

    foreach ($fixtures->getTestUrlInfo() as $providerName => $providerTestUrlInfo) {
      yield $providerName => [
        'providerName' => $providerName,
        'testUrlInfo' => $providerTestUrlInfo,
      ];

    }
  }

  /**
   * Tests that the resource fixtures are up-to-date.
   *
   * @param string $providerName
   *   The provider name.
   * @param array{urls: array<string, array{status: 'success|small_thumbnail|resource_exception', thumbnail_width?: int, exception_message?: string}>, query_parameters?: array<string, string>, replacements?: array<string, string>, missing_schemes?: string[]} $testUrlInfo
   *   The test data for this provider.
   *
   * @throws \ReflectionException
   */
  #[DataProvider('providersProvider')]
  public function testProviders(string $providerName, array $testUrlInfo): void {
    if (isset($testUrlInfo['query_parameters'])) {
      foreach ($testUrlInfo['query_parameters'] as $environmentVariable) {
        if (((string) getenv($environmentVariable)) === '') {
          $this->markTestSkipped("Missing '$environmentVariable' environment variable for provider $providerName");
        }
      }
    }

    $config = $this->prophesize(Config::class);
    $config->get('oembed_providers_url')->willReturn($this->providersUrl);
    $configFactory = $this->prophesize(ConfigFactoryInterface::class);
    $configFactory->get('media.settings')->willReturn($config->reveal());

    $keyValueStore = $this->prophesize(KeyValueStoreInterface::class);
    $keyValueFactory = $this->prophesize(KeyValueFactoryInterface::class);
    $keyValueFactory->get('media')->willReturn($keyValueStore->reveal());

    $loggerChannel = $this->prophesize(LoggerChannelInterface::class);
    $loggerChannelFactory = $this->prophesize(LoggerChannelFactoryInterface::class);
    $loggerChannelFactory->get('media')->willReturn($loggerChannel->reveal());

    $providerRepository = new ProviderRepository(
      $this->fixturesClient,
      $configFactory->reveal(),
      new Time(),
      $keyValueFactory->reveal(),
      $loggerChannelFactory->reveal(),
    );
    $urlResolver = new UrlResolver(
      $providerRepository,
      $this->prophesize(ResourceFetcherInterface::class)->reveal(),
      $this->fixturesClient,
      $this->prophesize(ModuleHandlerInterface::class)->reveal(),
      $this->prophesize(CacheBackendInterface::class)->reveal(),
    );

    // @todo Discover these automatically.
    $tweakClasses = [
      AddVersion::class,
    ];
    foreach ($tweakClasses as $tweakClass) {
      $reflectionClass = new \ReflectionClass($tweakClass);
      $attributes = $reflectionClass->getAttributes(OEmbedTweak::class);
      $attribute = $attributes[0]->newInstance();
      $this->assertInstanceOf(AttributeInterface::class, $attribute);
      $attribute->setClass($tweakClass);
      $tweaks[] = new $tweakClass([], $attribute->getId(), $attribute->get());
    }

    foreach (array_keys($testUrlInfo['urls']) as $testUrl) {
      $resourceUrl = $urlResolver->getResourceUrl($testUrl);
      $body = $this->handleFixture('GET', $resourceUrl, $testUrlInfo['query_parameters'] ?? [], $testUrlInfo['replacements'] ?? []);

      $before = json_decode($body, TRUE);
      if (isset($before['thumbnail_url'])) {
        $this->handleFixture('GET', $before['thumbnail_url']);
      }
      if (isset($before['author_url'])) {
        $this->handleFixture('HEAD', $before['author_url']);
      }

      $after = $before;
      foreach ($tweaks as $tweak) {
        if ($tweak->isApplicable($after, $testUrl)) {
          $after = $tweak->tweak($after, $testUrl);
        }
      }
      if (isset($after['thumbnail_url']) && (!isset($before['thumbnail_url']) || ($after['thumbnail_url'] !== $before['thumbnail_url']))) {
        $this->handleFixture('GET', $after['thumbnail_url']);
      }
      if (isset($after['author_url']) && (!isset($before['author_url']) || ($after['author_url'] !== $before['author_url']))) {
        $this->handleFixture('GET', $after['author_url']);
      }
    }
  }

  /**
   * Tests a given resource fixture.
   *
   * @param 'GET'|'HEAD' $method
   *   The HTTP method for the given URI.
   * @param string $uriString
   *   The fixture URI.
   * @param string[] $queryParameters
   *   An array of environment variable names, keyed by query parameter names,
   *   respectively. The given URI will be amended by adding the respective
   *   query parameters with the query parameter values being the values of the
   *   respective environment variables.
   * @param string[] $replacements
   *   An array of replacements for the remote response, where the keys are the
   *   strings to replace and the values are the respective replacements.
   */
  protected function handleFixture(string $method, string $uriString, array $queryParameters = [], array $replacements = []): string {
    $originalUri = new Uri($uriString);
    parse_str($originalUri->getQuery(), $queryParts);
    foreach ($queryParameters as $name => $environmentVariable) {
      $queryParts[$name] = getenv($environmentVariable);
    }
    $uri = $originalUri->withQuery(http_build_query($queryParts));

    try {
      $fixturesBody = (string) $this->fixturesClient->request($method, $originalUri)->getBody();
      // Use GET for the remote request even if the specified method is HEAD,
      // because some providers do not support HEAD requests.
      $remoteResponse = $this->remoteClient->get($uri);
    }
    catch (GuzzleException $exception) {
      $this->fail("Error fetching the body for $uriString: {$exception->getMessage()}");
    }

    $remoteBody = ($method === 'GET') ? (string) $remoteResponse->getBody() : '';
    if ($replacements) {
      $remoteBody = preg_replace(array_keys($replacements), array_values($replacements), $remoteBody);
    }

    if (!empty(getenv('OEMBED_TWEAK_UPDATE_FIXTURES'))) {
      $this->fixtures->updateFixture($originalUri, $remoteBody);
    }

    $this->assertEquals($fixturesBody, $remoteBody, "Outdated fixture for $uriString");
    return $fixturesBody;
  }

}
