<?php

declare(strict_types=1);

namespace Drupal\Tests\og_access\Functional;

use Drupal\Tests\BrowserTestBase;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\og\Og;
use Drupal\og\OgGroupAudienceHelperInterface;
use Drupal\og_access\OgAccess;

/**
 * Tests how OG Access handles group and group content visibility.
 *
 * @group og_access
 */
class OgAccessVisibilityTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'og',
    'og_access',
    'options',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * The bundle used for groups in this test.
   */
  protected NodeType $groupType;

  /**
   * The bundle used for group content in this test.
   */
  protected NodeType $groupContentType;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create a group bundle that has OG Access enabled.
    $this->groupType = NodeType::create([
      'type' => 'access_group',
      'name' => 'Access group',
    ]);

    $this->groupType->save();
    Og::groupTypeManager()->addGroup('node', $this->groupType->id());
    Og::createField(OgAccess::OG_ACCESS_FIELD, 'node', $this->groupType->id());

    // Create the group content bundle with access control.
    $this->groupContentType = NodeType::create([
      'type' => 'group_post',
      'name' => 'Group post',
    ]);

    $this->groupContentType->save();
    Og::createField(OgAccess::OG_ACCESS_CONTENT_FIELD, 'node', $this->groupContentType->id());

    // Turn the bundle into real OG content by adding the audience field.
    Og::createField(
      OgGroupAudienceHelperInterface::DEFAULT_FIELD,
      'node',
      $this->groupContentType->id(),
      [
        'field_storage_config' => [
          'settings' => [
            'target_type' => 'node',
          ],
        ],
        'field_config' => [
          'settings' => [
            'handler_settings' => [
              'target_bundles' => [
                $this->groupType->id() => $this->groupType->id(),
              ],
            ],
          ],
        ],
      ]
    );

    // Allow the node grants to kick in.
    node_access_rebuild();
  }

  /**
   * Ensures the module protects private groups and group content.
   */
  public function testGroupAndGroupContentVisibility(): void {
    $public_group = $this->createGroupNode('Public group', OgAccess::OG_ACCESS_PUBLIC);
    $private_group = $this->createGroupNode('Private group', OgAccess::OG_ACCESS_PRIVATE);

    // User that belongs to both groups.
    $member = $this->drupalCreateUser(['access content']);
    Og::createMembership($public_group, $member)->save();
    Og::createMembership($private_group, $member)->save();

    // Regular site visitor that is not a member of either group.
    $non_member = $this->drupalCreateUser(['access content']);

    // Public groups are accessible to everyone.
    $this->drupalLogin($non_member);
    $this->drupalGet($public_group->toUrl());
    $this->assertSession()->statusCodeEquals(200);

    // Private groups are restricted to their members.
    $this->drupalGet($private_group->toUrl());
    $this->assertSession()->statusCodeEquals(403);

    $this->drupalLogin($member);
    $this->drupalGet($private_group->toUrl());
    $this->assertSession()->statusCodeEquals(200);

    // Group content should respect both its own visibility and that
    // of its parent group.
    $public_content_in_private_group = $this->createGroupContentNode(
      'Public post in private group',
      $private_group,
      OgAccess::OG_ACCESS_PUBLIC
    );
    $private_content_in_public_group = $this->createGroupContentNode(
      'Private post in public group',
      $public_group,
      OgAccess::OG_ACCESS_PRIVATE
    );

    $this->drupalLogin($non_member);
    $this->drupalGet($public_content_in_private_group->toUrl());
    $this->assertSession()->statusCodeEquals(403);

    $this->drupalGet($private_content_in_public_group->toUrl());
    $this->assertSession()->statusCodeEquals(403);

    $this->drupalLogin($member);
    $this->drupalGet($public_content_in_private_group->toUrl());
    $this->assertSession()->statusCodeEquals(200);

    $this->drupalGet($private_content_in_public_group->toUrl());
    $this->assertSession()->statusCodeEquals(200);
  }

  /**
   * Creates a group node with the requested visibility.
   */
  protected function createGroupNode(string $title, int $visibility): Node {
    $group = Node::create([
      'type' => $this->groupType->id(),
      'title' => $title,
      'status' => 1,
    ]);
    $group->set(OgAccess::OG_ACCESS_FIELD, $visibility);
    $group->save();

    return $group;
  }

  /**
   * Creates group content linked to the given group.
   */
  protected function createGroupContentNode(string $title, Node $group, int $visibility): Node {
    $node = Node::create([
      'type' => $this->groupContentType->id(),
      'title' => $title,
      'status' => 1,
      'uid' => $this->rootUser->id(),
    ]);
    $node->set(OgGroupAudienceHelperInterface::DEFAULT_FIELD, [['target_id' => $group->id()]]);
    $node->set(OgAccess::OG_ACCESS_CONTENT_FIELD, $visibility);
    $node->save();

    return $node;
  }

}
