<?php

declare(strict_types=1);

namespace Drupal\Tests\og_access\Kernel;

use Drupal\Core\Field\EntityReferenceFieldItemListInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\node\NodeAccessControlHandlerInterface;
use Drupal\og\Og;
use Drupal\og\OgGroupAudienceHelperInterface;
use Drupal\og_access\OgAccess;
use Drupal\user\Entity\Role;
use Drupal\user\Entity\User;

/**
 * Tests OG Access node grant generation for group content.
 *
 * @group og_access
 */
class OgAccessNodeAccessRecordsTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'field',
    'node',
    'og',
    'og_access',
    'options',
    'system',
    'user',
  ];

  /**
   * The bundle used for groups in this test.
   */
  protected NodeType $groupType;

  /**
   * The bundle used for group content in this test.
   */
  protected NodeType $groupContentType;

  /**
   * A user who is not a member of the group.
   */
  protected User $nonMember;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installConfig(['og']);
    $this->installEntitySchema('node');
    $this->installEntitySchema('og_membership');
    $this->installEntitySchema('user');
    $this->installSchema('node', ['node_access']);
    $this->installSchema('system', ['sequences']);

    Role::create([
      'id' => 'authenticated',
      'label' => 'authenticated',
    ])->grantPermission('access content')->save();

    $this->nonMember = User::create([
      'name' => 'non-member',
    ]);
    $this->nonMember->save();

    // Create a group bundle that has OG Access enabled.
    $this->groupType = NodeType::create([
      'type' => 'access_group',
      'name' => 'Access group',
    ]);
    $this->groupType->save();
    Og::groupTypeManager()->addGroup('node', $this->groupType->id());
    Og::createField(OgAccess::OG_ACCESS_FIELD, 'node', $this->groupType->id());

    // Create the group content bundle with access control and an audience field
    // that references groups.
    $this->groupContentType = NodeType::create([
      'type' => 'group_post',
      'name' => 'Group post',
    ]);
    $this->groupContentType->save();
    Og::createField(OgAccess::OG_ACCESS_CONTENT_FIELD, 'node', $this->groupContentType->id());

    // Turn the bundle into real OG content by adding the audience field.
    Og::createField(
      OgGroupAudienceHelperInterface::DEFAULT_FIELD,
      'node',
      $this->groupContentType->id(),
      [
        'field_storage_config' => [
          'settings' => [
            'target_type' => 'node',
          ],
        ],
        'field_config' => [
          'settings' => [
            'handler_settings' => [
              'target_bundles' => [
                $this->groupType->id() => $this->groupType->id(),
              ],
            ],
          ],
        ],
      ]
    );
  }

  /**
   * Ensures node grants use raw og_audience target IDs.
   *
   * This prevents grant generation from being affected when audience field
   * referenced entities are filtered based on the current user.
   */
  public function testNodeAccessRecordsDoNotDependOnReferencedEntities(): void {
    $group = Node::create([
      'type' => $this->groupType->id(),
      'title' => 'Private group',
      'status' => 1,
    ]);
    $group->set(OgAccess::OG_ACCESS_FIELD, OgAccess::OG_ACCESS_PRIVATE);
    $group->save();

    $node = Node::create([
      'type' => $this->groupContentType->id(),
      'title' => 'Group post',
      'status' => 1,
      OgGroupAudienceHelperInterface::DEFAULT_FIELD => [['target_id' => $group->id()]],
      OgAccess::OG_ACCESS_CONTENT_FIELD => OgAccess::OG_ACCESS_PUBLIC,
    ]);
    $node->save();

    $this->container->get('current_user')->setAccount($this->nonMember);

    // Simulate audience referenced entities being filtered out for the current
    // user while the raw target IDs remain present. The implementation under
    // test must only use the raw field values.
    $audience = $this->createMock(EntityReferenceFieldItemListInterface::class);
    $audience->method('getValue')->willReturn([['target_id' => $group->id()]]);
    $audience->method('referencedEntities')->willReturn([]);
    $this->injectFieldItemList($node, OgGroupAudienceHelperInterface::DEFAULT_FIELD, $audience);

    $access_control_handler = \Drupal::entityTypeManager()->getAccessControlHandler('node');
    assert($access_control_handler instanceof NodeAccessControlHandlerInterface);
    $grants = $access_control_handler->acquireGrants($node);
    $this->assertTrue($this->hasGrant($grants, OgAccess::OG_ACCESS_REALM . ':node', (int) $group->id()));
    $this->assertFalse($this->hasGrant($grants, 'all', 0), 'Default "all" grant should not be used for private group content.');
  }

  /**
   * Injects a field item list into a node entity.
   */
  protected function injectFieldItemList(Node $node, string $field_name, EntityReferenceFieldItemListInterface $items): void {
    $reflection = new \ReflectionObject($node);
    $property = $reflection->getProperty('fields');
    $fields = $property->getValue($node);
    $fields[$field_name][LanguageInterface::LANGCODE_DEFAULT] = $items;
    $property->setValue($node, $fields);
  }

  /**
   * Checks whether a grant exists in a grant list.
   */
  protected function hasGrant(array $grants, string $realm, int $gid): bool {
    foreach ($grants as $grant) {
      if (($grant['realm'] ?? NULL) === $realm && (int) ($grant['gid'] ?? -1) === $gid) {
        return TRUE;
      }
    }
    return FALSE;
  }

}
