<?php

// @phpcs:ignoreFile cause of Drupal.Functions.MultiLineFunctionDeclaration.MissingTrailingComma (of commands options).

namespace Drupal\ohdear_integration\Drush\Commands;

use Consolidation\OutputFormatters\StructuredData\RowsOfFields;
use Drupal\Core\State\StateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\ohdear_integration\OhDearInfo;
use Drupal\ohdear_integration\OhDearSdkService;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * A Drush command file for ohdear_integration.
 */
final class OhdearIntegrationCommands extends DrushCommands {

  use StringTranslationTrait;

  /**
   * Constructs an OhdearIntegrationCommands object.
   */
  public function __construct(
    private readonly StateInterface $state,
    private readonly OhDearSdkService $ohdearSdk,
    private readonly OhDearInfo $ohdearInfo,
  ) {
    parent::__construct();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('state'),
      $container->get('ohdear_sdk'),
      $container->get('ohdear_integration.info'),
    );
  }

  /**
   * Prints the current maintenance status or starts/stops maintenance.
   *
   * @param array $options
   *   An associative array of options.
   *
   * @return \Consolidation\OutputFormatters\StructuredData\RowsOfFields
   *   A list of OhDear maintenance windows or activity description.
   */
  #[CLI\Command(name: 'ohdear:maintenance', aliases: [
    'ohdear-maintenance',
    'ohdear-m',
    'ohdear:m',
  ])]
  #[CLI\Option(name: 'start', description: 'Starts the maintenance period in OhDear app for current site.')]
  #[CLI\Option(name: 'stop', description: 'Stops the maintenance  period in OhDear app for current site.')]
  #[CLI\Option(name: 'with-drupal', description: 'Puts drupal in or out of maintenance mode.')]
  #[CLI\Option(name: 'time-length', description: 'Time length for the maintenance period in seconds (default: 1 hour).')]
  #[CLI\Option(name: 'label', description: 'Label for the maintenance period (eg: Deploy XYZ upgrade).')]
  #[CLI\Option(name: 'number', description: 'Number of maintenance  periods to show (default: 5).')]
  #[CLI\Usage(name: 'ohdear:maintenance', description: 'Prints the current maintenance status or starts/stops maintenance and last 5 (default number) maintenance periods.')]
  #[CLI\Usage(name: 'ohdear:maintenance --start --with-drupal', description: 'Starts the maintenance period in OhDear app and puts drupal in (or out of) maintenance mode (for --stop flag) .')]
  #[CLI\Usage(name: 'ohdear:maintenance --start --label="Update module XYZ" --time-length=900', description: 'Starts the 15 minutes maintenance period in OhDear app named "Update module XYZ".')]
  #[CLI\FieldLabels(labels: [
    'id' => 'Id',
    'name' => 'Label',
    'starts_at' => 'Starts at',
    'ends_at' => 'Ends at',
  ])]
  #[CLI\DefaultTableFields(fields: ['id', 'name', 'starts_at', 'ends_at'])]
  #[CLI\FilterDefaultField(field: 'id')]
  public function maintenance(array $options = [
    'start' => NULL,
    'stop' => NULL,
    'with-drupal' => NULL,
    'time-length' => 3600,
    'number' => 5,
    'label' => NULL,
  ]) :void {
    $ohDear = $this->ohdearSdk->getOhDear();
    // Only the current site can be put in maintenance mode, that is why we get
    // monitor id from config or env variable.
    $monitorId = $this->ohdearSdk->getMonitorId();
    $monitor = $ohDear->monitor($monitorId);
    // @see https://ohdear.app/docs/integrations/the-oh-dear-php-sdk#maintenance-windows
    if ($options['start'] && $options['stop']) {
      $this->logger->warning('Only one of --start or --stop flags can be used.');
      return;
    }
    elseif ($options['start']) {
      $time_length = $options['time-length'] ?? 3600;
      $maintenance_window = $ohDear->startMaintenancePeriod($monitorId, $time_length, $options['label'] ?? NULL);
      if ($options['with-drupal']) {
        $this->state->set('system.maintenance_mode', 1);
        $this->output()->writeln($this->t('Drupal put in maintenance mode. Do not forget to turn off maintenance mode with --with-drupal flag when maintenance period stops.'));
      }
      $this->output()->writeln($this->t('Maintenance period @id started for @site. It ends at @date.', [
        '@id' => $maintenance_window->id,
        '@site' => $monitor->url,
        '@date' => $maintenance_window->endsAt ?? '',
      ]));
    }
    elseif ($options['stop']) {
      $ohDear->stopMaintenancePeriod($monitorId);
      if ($options['with-drupal']) {
        $this->state->set('system.maintenance_mode', 0);
      }
      $this->output()->writeln($this->t('Maintenance period stopped for @site.', [
        '@site' => $monitor->url,
      ]));
    }
    $maintenanceWindows = $this->getParsedMaintenancePeriods($monitorId, $options['number']);
    $this->io()->table(['Id', 'Label', 'StartAt', 'EndAt', 'Active'], $maintenanceWindows);
  }

  /**
   * Gets maintenance period for specific site.
   *
   * @param int|null $monitorId
   *   Monitor id.
   * @param int $num
   *   Number of maintenance windows.
   *
   * @return array
   *   Array of last maintenance periods.
   *
   * @throws \Exception
   */
  protected function getParsedMaintenancePeriods(?int $monitorId = NULL, int $num = 5) {
    $maintenancePeriods = $this->ohdearInfo->getMaintenancePeriods($monitorId);
    $maintenancePeriods = array_slice(array_reverse($maintenancePeriods), 0, $num, TRUE);
    if (empty($maintenancePeriods)) {
      $this->logger->warning('No maintenance periods found.');
    }
    $rows = [];
    foreach ($maintenancePeriods as $period) {
      $rows[] = [
        'id' => $period->id,
        'name' => $period->name,
        'starts_at' => $period->startsAt,
        'ends_at' => $period->endsAt ?? '',
        'active' => strtotime($period->endsAt) > time() ? 'Yes' : 'No',
      ];
    }
    return $rows;
  }

  /**
   * Prints current monitor id and url.
   *
   * @return \Consolidation\OutputFormatters\StructuredData\RowsOfFields
   *   A list of OhDear checks.
   */
  #[CLI\Command(name: 'ohdear:info', aliases: [
    'ohdear-info',
    'ohdear-i',
    'ohdear:i',
  ])]
  #[CLI\Argument(name: 'monitorId', description: 'Optional monitorId parameter. By default it takes the value provided by configuration but this allows to check other websites as well.')]
  #[CLI\Option(name: 'checks', description: 'Get a list of health checks for site.')]
  #[CLI\Option(name: 'list-sites', description: 'Print all available site ids.')]
  #[CLI\Usage(name: 'ohdear_integration:info', description: 'Prints current site id and url.')]
  #[CLI\FieldLabels(labels: [
    'id' => 'Id',
    'type' => 'Type',
    'label' => 'Label',
    'enabled' => 'Enabled',
    'latest_run_ended_at' => 'Latest run ended at',
    'latest_run_result' => 'Latest run result',
    'summary' => 'Summary',
  ])]
  #[CLI\DefaultTableFields(fields: [
    'id',
    'label',
    'latest_run_ended_at',
    'latest_run_result',
    'summary',
    'enabled',
  ])]
  #[CLI\FilterDefaultField(field: 'id')]
  public function siteInfo(int $monitorId = NULL, array $options = [
    'checks' => NULL,
    'list-sites' => NULL,
  ]) {
    $ohDear = $this->ohdearSdk->getOhDear();
    $monitorId = $this->ohdearInfo->getProvidedOrCurrentMonitorId($monitorId);
    if (!$monitorId) {
      $this->logger()->error('Monitor id not provided!');
      return NULL;
    }
    $monitor = $ohDear->monitor($monitorId);
    $this->output()->writeln($this->t('Site ID: @id, Url: @url', [
      '@id' => (string) $monitor->id,
      '@url' => $monitor->url,
    ]));
    $this->output()->writeln($this->t('Summarized check results: @results (@date).', [
      '@results' => $monitor->summarizedCheckResult ?? 'unknown',
      '@date' => $monitor->latestRunDate ?? '',
    ]));
    if (isset($options['checks'])) {
      $checks = [];
      foreach ($monitor->checks as $check) {
        unset($check['settings']);
        $check['enabled'] = $check['enabled'] ? 'On' : 'Off';
        $check['active_snooze'] = $check['active_snooze'] ? 'Yes' : 'No';
        $checks[] = $check;
      }
      $this->io()->table(array_keys($check), $checks);
      // return new RowsOfFields($checks);
    }
    if (isset($options['list-sites'])) {
      $monitors = $this->ohdearInfo->listMonitors();
      if (!is_array($monitors)) {
        $monitors = iterator_to_array($monitors);
      }
      $all = array_map(fn ($monitor) => [
        'id' => (string) $monitor->id,
        'label' => $monitor->label ?? $monitor->url,
      ], $monitors);
      $this->io()->table(['Id', 'Label'], $all);
    }
    return NULL;
  }

  /**
   * Get broken links for current site.
   *
   * @param int|null $monitorId
   *   The id of the monitor or empty for current website.
   *
   * @return \Consolidation\OutputFormatters\StructuredData\RowsOfFields
   *   A list of broken links.
   */
  #[CLI\Command(name: 'ohdear:broken-links', aliases: [
    'ohdear-broken-links',
    'ohdear-bl',
    'ohdear:bl',
  ])]
  #[CLI\Argument(name: 'monitorId', description: 'Optional monitorId parameter. By default it takes the value provided by configuration but this allows to check other websites as well.')]
  #[CLI\Usage(name: 'ohdear:broken-links', description: 'Get broken links for current site.')]
  #[CLI\FieldLabels(labels: [
    'crawledUrl' => 'Crawled URL',
    'statusCode' => 'Status Code',
    'foundOnUrl' => 'Found On URL',
  ])]
  #[CLI\DefaultTableFields(fields: ['crawledUrl', ',statusCode', 'foundOnUrl'])]
  #[CLI\FilterDefaultField(field: 'crawledUrl')]
  public function getBrokenLinks(int $monitorId = NULL) {
    $parsedBrokenLinks = $this->ohdearInfo->getParsedBrokenLinks($monitorId);
    return new RowsOfFields($parsedBrokenLinks);
  }

  /**
   * Get uptime for site. It shows last 7 days by default.
   *
   * @param int $monitorId
   *   Monitor id.
   * @param array $options
   *   List of options.
   *
   * @return \Consolidation\OutputFormatters\StructuredData\RowsOfFields
   *   A list of broken links.
   */
  #[CLI\Command(name: 'ohdear:uptime', aliases: [
    'ohdear-uptime',
    'ohdear-u',
    'ohdear:u',
  ])]
  #[CLI\Argument(name: 'monitorId', description: 'Optional monitorId parameter. By default it takes the value provided by configuration but this allows to check other websites as well.')]
  #[CLI\Option(name: 'from', description: 'Date from in `Y-m-d H:i:s` format.')]
  #[CLI\Option(name: 'to', description: 'End date in `Y-m-d H:i:s` format.')]
  #[CLI\Option(name: 'split', description: 'Time unit to split the interval (day, week, month).')]
  #[CLI\Usage(name: 'ohdear:uptime', description: 'Get uptime for last 7 days (default).')]
  #[CLI\Usage(name: 'ohdear:uptime  --split month --from 2022-01-01 00:00:00', description: 'Shows monthly uptime for current monitor from 1. 1. 2022.')]
  #[CLI\FieldLabels(labels: [
    'datetime' => 'DateTime',
    'percentage' => 'Uptime Percentage',
  ])]
  #[CLI\DefaultTableFields(fields: ['datetime', ',percentage'])]
  #[CLI\FilterDefaultField(field: 'datetime')]
  public function getUptime(int $monitorId = NULL, array $options = [
    'from' => NULL,
    'to' => NULL,
    'split' => 'day',
  ]) {
    $uptimes = $this->ohdearInfo->getParsedUptime($monitorId, $options['from'] ?: date('Y-m-d H:i:s', strtotime('-7 days')), $options['to'] ?: date('Y-m-d H:i:s', strtotime('now')), $options['split']);
    return new RowsOfFields($uptimes);
  }

}
