<?php

namespace Drupal\ohdear_integration;

use OhDear\PhpSdk\Dto\Monitor;
use OhDear\PhpSdk\Enums\UptimeSplit;

/**
 * Service to retrieve info from OhDear api.
 */
class OhDearInfo {

  use DateFormatValidationTrait;

  /**
   * The OhDear php service development kit object.
   *
   * @var \Drupal\ohdear_integration\OhDearSdkService
   */
  protected OhDearSdkService $ohDearSdk;

  /**
   * Construct method.
   *
   * @param \Drupal\ohdear_integration\OhDearSdkService $ohDearSdkService
   *   OhDear php sdk.
   */
  public function __construct(OhDearSdkService $ohDearSdkService) {
    $this->ohDearSdk = $ohDearSdkService;
  }

  /**
   * Helper method to get monitor id if not provided.
   *
   * @param int|null $monitorId
   *   Monitor id or null if configured monitor is wanted.
   *
   * @return int
   *   Will return monitor id integer.
   *
   * @throws \Exception
   */
  public function getProvidedOrCurrentMonitorId(?int $monitorId = NULL) {
    $monitorId == 0 && $monitorId = NULL;
    return $monitorId ?? $this->ohDearSdk->getMonitorId();
  }

  /**
   * Get a list of monitors available with current OhDear API key.
   *
   * @return iterable<int, Monitor>
   *   Array of monitors with id and url properties.
   *
   * @throws \Exception
   *   When API key is missing or invalid.
   */
  public function listMonitors() : iterable {
    return $this->ohDearSdk->getOhDear()->monitors();
  }

  /**
   * Get site label of monitor.
   *
   * @param int|null $monitorId
   *   Monitor id or falsy for current monitor.
   *
   * @return string
   *   Return monitor label or url.
   */
  public function getSiteLabel(?int $monitorId) :string {
    $monitorId = $this->getProvidedOrCurrentMonitorId($monitorId);
    return $this->getMonitor($monitorId)->label ?? $this->getMonitor($monitorId)->url;
  }

  /**
   * Get an option list of sites.
   *
   * @return array
   *   Array of site ids as keys and labels as values.
   */
  public function prettySiteList() : array {
    return array_map(fn ($monitor) => [$monitor->id => $monitor->label ?? $monitor->url], $this->listMonitors());
  }

  /**
   * Get uptimes - default value is last 30 days.
   *
   * @param int $monitorId
   *   The monitor id.
   * @param string $from
   *   Start date with 'YmdHis' format.
   * @param string $to
   *   End date with 'YmdHis' format.
   * @param string $split
   *   Time interval (hour, day, month, year).
   *
   * @return \OhDear\PhpSdk\Dto\Uptime[]
   *   Array of uptimes.
   */
  public function getUptime(?int $monitorId = NULL, ?string $from = NULL, ?string $to = NULL, string $split = 'day') : array {
    if (!$from) {
      $from = date('Y-m-d H:i:s', strtotime('-30 days'));
    }
    if (!$to || $to < $from) {
      $to = date('Y-m-d H:i:s');
    }

    return $this->ohDearSdk->getOhDear()->uptime($monitorId, $from, $to, UptimeSplit::tryFrom($split));
  }

  /**
   * Get parsed uptimes - default value is last 7 days.
   *
   * @param int|null $monitorId
   *   The monitor id to check.
   * @param string|null $from
   *   Start date with 'Y-m-d H:i:s' format.
   * @param string|null $to
   *   End date with 'Y-m-d H:i:s' format.
   * @param string $split
   *   Time interval (hour, day, month, year).
   *
   * @return array
   *   Array of uptimes with only relevant properties.
   *
   * @throws \Exception
   */
  public function getParsedUptime(?int $monitorId = NULL, ?string $from = NULL, ?string $to = NULL, string $split = 'day') : array {
    $monitorId = $this->getProvidedOrCurrentMonitorId($monitorId);
    if (isset($from) && !$this->isValidDateFormat($from)) {
      throw new \InvalidArgumentException('The start date parameter is not in valid format. Use `Y-m-d H:i:s` format.');
    }
    if (isset($to) && !$this->isValidDateFormat($to)) {
      throw new \InvalidArgumentException('The end date parameter is not in valid format. Use `Y-m-d H:i:s` format.');
    }
    $uptimes = $this->getUptime($monitorId, $from ?: date('Y-m-d H:i:s', strtotime('-7 days')), $to ?: date('Y-m-d H:i:s', strtotime('now')), $split);
    return array_map(fn ($uptime) => [
      'datetime' => $split == 'hour' ? $uptime->datetime : date('Y-m-d H:i:s', strtotime($uptime->datetime)),
      'percentage' => $uptime->uptimePercentage,
    ], $uptimes);
  }

  /**
   * Get broken links for monitor.
   *
   * @return iterable<int, BrokenLink>
   *   Array of broken links.
   */
  public function getBrokenLinks(?int $monitorId = NULL) : iterable {
    return $this->ohDearSdk->getOhDear()->brokenLinks($monitorId);
  }

  /**
   * Get parsed broken links for monitor.
   *
   * @param int|null $monitorId
   *   The monitor id to check.
   *
   * @return array
   *   Array of broken links with only relevant properties.
   */
  public function getParsedBrokenLinks(?int $monitorId = NULL) : array {
    $monitorId = $this->getProvidedOrCurrentMonitorId($monitorId);
    $brokenLinks = $this->getBrokenLinks($monitorId);
    if (!is_array($brokenLinks)) {
      $brokenLinks = iterator_to_array($brokenLinks);
    }
    return array_map(fn ($link) => [
      'crawledUrl' => $link->crawledUrl ?? '',
      'statusCode' => $link->statusCode ?? '',
      'foundOnUrl' => $link->foundOnUrl ?? '',
    ], $brokenLinks);
  }

  /**
   * Get OhDear checks for monitor.
   *
   * @param int|null $monitorId
   *   The monitor id to check.
   *
   * @return \OhDear\PhpSdk\Resources\Check[]
   *   Array of checks.
   */
  public function getChecks(?int $monitorId = NULL) {
    return $this->getMonitor($monitorId)?->checks ?? [];
  }

  /**
   * Get monitor helper method.
   *
   * Returns the configured monitor or the one provided by monitorId argument
   * which can be different but should be accessible with the configured OhDear
   * API key.
   *
   * @param int|null $monitorId
   *   Monitor id or null if you want to get the default one.
   *
   * @return \OhDear\PhpSdk\Dto\Monitor
   *   The OhDear php sdk monitor object instance.
   */
  protected function getMonitor(?int $monitorId = NULL): Monitor {
    return $monitorId ? $this->ohDearSdk->getOhDear()->monitor($monitorId) : $this->ohDearSdk->getMonitor();
  }

  /**
   * Get maintenance periods for monitor.
   *
   * @param int|null $monitorId
   *   The monitor id to check.
   *
   * @return array
   *   Array of maintenance periods.
   *
   * @throws \Exception
   */
  public function getMaintenancePeriods(?int $monitorId = NULL) {
    $monitorId = $this->getProvidedOrCurrentMonitorId($monitorId);
    $ohDear = $this->ohDearSdk->getOhDear();
    $maintenancePeriods = $ohDear->maintenancePeriods($monitorId);
    $maintenanceWindows = [];
    foreach ($maintenancePeriods as $period) {
      $maintenanceWindows[] = $period;
    }
    return $maintenanceWindows;
  }

}
