<?php

namespace Drupal\franceconnect\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for FranceConnect claims.
 *
 * This form allows the site administrator to select the claims
 * that the FranceConnect client is authorized to access, based on
 * the information provided with DataPass.
 *
 * https://docs.partenaires.franceconnect.gouv.fr/fs/devenir-fs/projet-datapass/
 */
class FranceConnectClaimsConfigForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'franceconnect_claims_config_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['franceconnect.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {

    $form['help'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . $this->t('Configurez ici les claims à exposer et leur scope associé. Laissez vide pour exclure un claim. Un claim peut être associé à un seul scope, mais un scope peut être utilisé pour plusieurs claims. Cette configuration sera ensuite appliquée dans la configuration du plugin <strong>FranceConnect</strong> du module <strong>OpenID Connect</strong>. <br> Vous pouvez consulter la liste des claims disponibles <a href="https://docs.partenaires.franceconnect.gouv.fr/fs/fs-technique/fs-technique-scope-fc/#correspondance-entre-scope-et-claims" target="_blank">ici</a>.') . '</p>',
    ];

    $config = $this->config('franceconnect.settings');

    // See https://docs.partenaires.franceconnect.gouv.fr/fs/fs-technique/fs-technique-scope-fc/
    $scope_claim_map = [
      'openid' => ['sub'],
      'gender' => ['gender'],
      'birthdate' => ['birthdate'],
      'birthcountry' => ['birthcountry'],
      'birthplace' => ['birthplace'],
      'given_name' => ['given_name', 'given_name_array'],
      'family_name' => ['family_name'],
      'email' => ['email'],
      'preferred_username' => ['preferred_username'],
      'profile' => ['family_name', 'given_name', 'preferred_username', 'gender', 'birthdate'],
      'birth' => ['birthplace', 'birthcountry'],
      'identite_pivot' => ['given_name', 'family_name', 'birthdate', 'gender', 'birthplace', 'birthcountry'],
      'rnipp_given_name' => ['given_name', 'rnipp_given_name'],
      'rnipp_family_name' => ['family_name', 'rnipp_family_name'],
      'rnipp_gender' => ['gender', 'rnipp_gender'],
      'rnipp_birthcountry' => ['birthcountry', 'rnipp_birthcountry'],
      'rnipp_birthplace' => ['birthplace', 'rnipp_birthplace'],
      'rnipp_birthdate' => ['birthdate', 'rnipp_birthdate'],
      'rnipp_profile' => ['given_name', 'family_name', 'birthdate', 'gender', 'preferred_username', 'rnipp_given_name', 'rnipp_family_name', 'rnipp_birthdate', 'rnipp_gender'],
      'rnipp_birth' => ['birthplace', 'birthcountry', 'rnipp_birthplace', 'rnipp_birthcountry'],
      'rnipp_identite_pivot' => ['given_name', 'family_name', 'birthdate', 'gender', 'birthplace', 'birthcountry', 'rnipp_given_name', 'rnipp_family_name', 'rnipp_birthdate', 'rnipp_gender', 'rnipp_birthplace', 'rnipp_birthcountry'],
    ];

    $unique_claims = [];
    foreach ($scope_claim_map as $scope => $claims_list) {
      foreach ($claims_list as $claim) {
        if (!isset($unique_claims[$claim])) {
          $unique_claims[$claim] = [];
        }
        if (!in_array($scope, $unique_claims[$claim])) {
          $unique_claims[$claim][] = $scope;
        }
      }
    }

    ksort($unique_claims);
    $saved_claims = $config->get('claims') ?? [];

    $form['claims_container'] = [
      '#type' => 'container',
      '#tree' => TRUE,
    ];

    foreach ($unique_claims as $claim => $scopes) {
      $form['claims_container'][$claim] = [
        '#type' => 'select',
        '#title' => $this->t('@claim', ['@claim' => $claim]),
        '#options' => array_combine($scopes, $scopes),
        '#empty_option' => $this->t('- Select a scope -'),
        '#default_value' => $saved_claims[$claim] ?? '',
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);

    $claims_config = $form_state->getValue('claims_container');
    $claims_config = array_filter($claims_config);

    $this->config('franceconnect.settings')
      ->set('claims', $claims_config)
      ->save();

    $this->messenger()->addStatus($this->t('Claims configuration saved.'));
  }

}
