<?php

namespace Drupal\franceconnect\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\Core\Url;
use Drupal\franceconnect\Services\FranceConnectService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Response;

/**
 * FranceConnect controller.
 */
class FranceConnectController extends ControllerBase {

  public function __construct(FranceConnectService $franceconnectService, PrivateTempStoreFactory $privateTempstoreFactory) {
    $this->franceConnectService = $franceconnectService;
    $this->privateTempStore = $privateTempstoreFactory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('franceconnect.service'),
      $container->get('tempstore.private')
    );
  }

  /**
   * Logout callback.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   Redirecting the user after logout.
   */
  public function userLogout() {

    $url = NULL;

    $franceconnectPrivateTempStore = $this->privateTempStore->get('franceconnect');
    $tokenData = $franceconnectPrivateTempStore->get('franceconnect_token');

    if (!empty($tokenData)) {
      try {
        $url = $this->franceConnectService->logoutUrl($tokenData);
      }
      catch (\Exception $exception) {
        \Drupal::logger('franceconnect')->error($exception->getMessage());
      }
    }

    user_logout();

    if (empty($url)) {
      return $this->redirect('<front>');
    }
    else {
      return $this->getExternalResponse($url);
    }
  }

  /**
   * Logout callback action.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   Redirect to the homepage.
   */
  public function logoutCallback() {
    $franceconnectPrivateTempStore = $this->privateTempStore->get('franceconnect');
    $franceconnectPrivateTempStore->delete('franceconnect_token');

    $url = Url::fromRoute('<front>')->toString();
    return new RedirectResponse($url);
  }

  /**
   * Redirect to external Url.
   *
   * @param string $externalUrl
   *   External Url.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   Response object.
   */
  private function getExternalResponse($externalUrl) {
    $response = new Response('', 302, []);
    $response->headers->set('Location', $externalUrl);

    return $response;
  }

}
