<?php

namespace Drupal\Tests\franceconnect\Unit;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\franceconnect\Services\FranceConnectService;
use Drupal\openid_connect\OpenIDConnectStateTokenInterface;
use Drupal\Tests\UnitTestCase;
use Firebase\JWT\JWT;
use GuzzleHttp\ClientInterface;

/**
 * Unit testing of the FranceConnectService class.
 */
class FranceConnectServiceTest extends UnitTestCase {
  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;
  /**
   * The OpenID state token service.
   *
   * @var \Drupal\openid_connect\OpenIDConnectStateTokenInterface
   */
  protected $stateToken;
  /**
   * The HTTP client to fetch the feed data with.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;
  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;
  /**
   * The logger.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;
  /**
   * The FranceConnect service.
   *
   * Provides helper methods for handling FranceConnect authentication and data.
   *
   * @var \Drupal\franceconnect\Services\FranceConnectService
   */
  protected $franceConnectService;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->stateToken = $this->createMock(OpenIDConnectStateTokenInterface::class);
    $this->httpClient = $this->createMock(ClientInterface::class);
    $this->loggerFactory = $this->createMock(LoggerChannelFactoryInterface::class);
    $this->logger = $this->createMock(LoggerChannelInterface::class);

    $this->loggerFactory->method('get')->willReturn($this->logger);

    $this->franceConnectService = new FranceConnectService(
      $this->configFactory,
      $this->stateToken,
      $this->httpClient,
      $this->loggerFactory
    );
  }

  /**
   * Tests token validation with a valid ES256 signature.
   *
   * This test generates a JWT using a private EC key and verifies that the
   * FranceConnectService::validateToken method correctly validates the
   * signature against the provided JWK set. The method is expected to return
   * a decoded object containing the payload data.
   */
  public function testValidateTokenWithValidSignature() {

    $privateKey = <<<EOD
-----BEGIN PRIVATE KEY-----
MIGHAgEAMBMGByqGSM49AgEGCCqGSM49AwEHBG0wawIBAQQgwJBIcSfzklLbJRWv
3F0httvZdh1mhKOO5/woR5zB2V2hRANCAAScKhYCgVgc1YOxFgMNkz5iTfkE+8KU
PL3j+KykRBH1xqLFO335A8V6i9R2BAMtLw//ZTHzrdjm1r18FM3ORr8h
-----END PRIVATE KEY-----
EOD;

    $jwkSet = [
      'keys' => [
        [
          "kty" => "EC",
          "use" => "sig",
          "alg" => "ES256",
          "kid" => "661482cb-4a33-497c-9ce2-c3b40dd8656a",
          "crv" => "P-256",
          "x" => "nCoWAoFYHNWDsRYDDZM-Yk35BPvClDy94_ispEQR9cY",
          "y" => "osU7ffkDxXqL1HYEAy0vD_9lMfOt2ObWvXwUzc5GvyE",
        ],
      ],
    ];

    $payload = ['sub' => '1234567890', 'name' => 'John Doe', 'iat' => time()];
    $jwt = JWT::encode($payload, $privateKey, 'ES256', "661482cb-4a33-497c-9ce2-c3b40dd8656a");
    $result = $this->franceConnectService->validateToken($jwt, $jwkSet);

    $this->assertIsObject($result);
    $this->assertEquals('1234567890', $result->sub);
  }

  /**
   * Tests token validation with an invalid signature.
   *
   * This test passes a fake JWT with a deliberately invalid signature to
   * FranceConnectService::validateToken. The test asserts that the method
   * returns null and logs an error, indicating proper handling of
   * signature verification failure.
   */
  public function testValidateTokenWithInvalidSignature() {
    $fakeJwt = 'eyJhbGciOiJSUzI1NiIsInR5cCI6IkpXVCJ9.eyJzdWIiOiIxMjMiLCJuYW1lIjoiSmVhbiJ9.fake-signature';

    $jwkSet = [
      'keys' => [
        [
          "kty" => "EC",
          "use" => "sig",
          "alg" => "ES256",
          "kid" => "661482cb-4a33-497c-9ce2-c3b40dd8656a",
          "crv" => "P-256",
          "x" => "ZaXOFFWqBFDsoGsJDUtdCbyaFJHlt_JUM7OmhOHaq14",
          "y" => "Lstaqxqtrr-hRGRlv714zHQWV47nF-JUz_bqTFIaOu8",
        ],
      ],
    ];

    $this->logger->expects($this->once())
      ->method('error')
      ->with($this->stringContains('empty'));

    $result = $this->franceConnectService->validateToken($fakeJwt, $jwkSet);
    $this->assertNull($result);
  }

}
