<?php

namespace Drupal\franceconnect\Services;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Url;
use Drupal\openid_connect\OpenIDConnectStateTokenInterface;
use Firebase\JWT\JWK;
use Firebase\JWT\JWT;
use GuzzleHttp\ClientInterface;

/**
 * Provides helper methods for handling FranceConnect authentication and data.
 */
class FranceConnectService {
  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;
  /**
   * The OpenID state token service.
   *
   * @var \Drupal\openid_connect\OpenIDConnectStateTokenInterface
   */
  protected $stateToken;
  /**
   * The HTTP client to fetch the feed data with.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;
  /**
   * The logger factory used for logging.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  public function __construct(ConfigFactoryInterface $configFactory, OpenIDConnectStateTokenInterface $state_token, ClientInterface $http_client, LoggerChannelFactoryInterface $logger_factory) {
    $this->configFactory = $configFactory;
    $this->stateToken = $state_token;
    $this->httpClient = $http_client;
    $this->loggerFactory = $logger_factory;
  }

  /**
   * Get logout url.
   *
   * @return string|null
   *   Logout url.
   */
  public function logoutUrl(array $token_data) {
    $state_token = $this->stateToken->create();
    if (!empty($token_data['id_token'])) {
      $config = $this->configFactory->get('openid_connect.settings.franceconnect');
      $session_endpoint = $config->get('settings')['session_end_endpoint'];

      $redirect_uri = Url::fromRoute('franceconnect.logout_callback', [], ['absolute' => TRUE])->toString();

      return Url::fromUri($session_endpoint, [
        'query' => [
          'id_token_hint' => $token_data['id_token'],
          'state' => $state_token,
          'post_logout_redirect_uri' => $redirect_uri,
        ],
      ])->toString();
    }

    return NULL;
  }

  /**
   * Retrieves the JWKS keys from the FranceConnect endpoint.
   *
   * This function fetches the JSON Web Key Set (JWKS) from the
   * FranceConnect public endpoint.
   */
  public function getJwksKeys($jwksUri) {
    $response = $this->httpClient->get($jwksUri);

    if ($response->getStatusCode() === 200) {
      $body = $response->getBody()->getContents();
      $keySet = json_decode($body, TRUE);
      return $keySet;
    }
    else {
      throw new \Exception(t('Erreur lors de la récupération du JWKS : ') . $response->getStatusCode());
    }
  }

  /**
   * Uses the JWKS keys to validate the signature of a token.
   */
  public function validateToken($jwt, $keySet) {
    try {
      $decoded = JWT::decode($jwt, JWK::parseKeySet($keySet));
      return $decoded;
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('franceconnect')->error($e->getMessage());
      return NULL;
    }
  }

}
