<?php

namespace Drupal\oidc_mcpf\Form;

use Drupal\Component\Plugin\PluginManagerInterface;
use Drupal\Core\Asset\LibraryDiscoveryInterface;
use Drupal\Core\Condition\ConditionInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\SubformState;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form to edit the module settings.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * The condition manager.
   *
   * @var \Drupal\Component\Plugin\PluginManagerInterface
   */
  protected PluginManagerInterface $conditionManager;

  /**
   * The library discovery service.
   *
   * @var \Drupal\Core\Asset\LibraryDiscoveryInterface
   */
  protected LibraryDiscoveryInterface $libraryDiscovery;

  /**
   * The condition plugin instances.
   *
   * @var \Drupal\Core\Condition\ConditionInterface[]
   */
  protected array $conditionPlugins = [];

  /**
   * Class constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   * @param \Drupal\Component\Plugin\PluginManagerInterface $condition_manager
   *   The condition manager.
   * @param \Drupal\Core\Asset\LibraryDiscoveryInterface $library_discovery
   *   The library discovery service.
   */
  public function __construct(ConfigFactoryInterface $config_factory, PluginManagerInterface $condition_manager, LibraryDiscoveryInterface $library_discovery) {
    parent::__construct($config_factory);

    $this->conditionManager = $condition_manager;
    $this->libraryDiscovery = $library_discovery;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('config.factory'),
      $container->get('plugin.manager.condition'),
      $container->get('library.discovery')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['oidc_mcpf.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'oidc_mcpf_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('oidc_mcpf.settings');

    $form['general'] = [
      '#type' => 'details',
      '#title' => $this->t('General'),
      '#open' => TRUE,
    ];

    $form['general']['environment'] = [
      '#type' => 'radios',
      '#title' => $this->t('Environment'),
      '#options' => [
        'test' => $this->t('Test'),
        'production' => $this->t('Production'),
      ],
      '#default_value' => $config->get('environment'),
      '#required' => TRUE,
    ];

    $form['toolbar'] = [
      '#type' => 'details',
      '#title' => $this->t('Citizen profile toolbar'),
      '#open' => TRUE,
      '#tree' => TRUE,
    ];

    $form['toolbar']['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enabled'),
      '#default_value' => (int) $config->get('toolbar.enabled'),
    ];

    $form['toolbar']['wrapper'] = [
      '#type' => 'container',
      '#parents' => ['toolbar'],
      '#states' => [
        'visible' => [
          ':checkbox[name="toolbar[enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['toolbar']['wrapper']['widget_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Widget ID'),
      '#default_value' => $config->get('toolbar.widget_id'),
      '#states' => [
        'required' => [
          ':checkbox[name="toolbar[enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['toolbar']['wrapper']['language_switcher'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Language switcher'),
      '#description' => $this->t('Must be enabled in the toolbar configuration of the governement as well.'),
      '#default_value' => (int) $config->get('toolbar.language_switcher'),
    ];

    // Store the used conditions in the form state.
    if (!$form_state->has('toolbar_conditions')) {
      $conditions = $config->get('toolbar.conditions') ?: [];

      if (!empty($conditions)) {
        $conditions = array_intersect_key($conditions, $this->conditionManager->getDefinitions());
      }

      $form_state->set('toolbar_conditions', $conditions);
    }

    $form['toolbar']['wrapper']['conditions'] = [
      '#prefix' => '<div id="oidc-mcpf-toolbar-conditions">',
      '#suffix' => '</div>',
    ];

    $form['toolbar']['wrapper']['conditions']['enabled'] = [
      '#type' => 'details',
      '#title' => $this->t('Visibility conditions'),
      '#open' => TRUE,
      '#access' => !empty($form_state->get('toolbar_conditions')),
    ];

    $form['toolbar']['wrapper']['conditions']['enabled']['_tabs'] = [
      '#type' => 'vertical_tabs',
    ];

    $options = [];
    foreach ($this->conditionManager->getDefinitions() as $plugin_id => $definition) {
      // Add as option if not yet added.
      if (!$form_state->has(['toolbar_conditions', $plugin_id])) {
        $options[$plugin_id] = $definition['label'];
        continue;
      }

      // Get and configure the plugin.
      $plugin = $this->getConditionPlugin($plugin_id);
      $plugin->setConfiguration(
        $form_state->get(['toolbar_conditions', $plugin_id])
      );

      // Create the element.
      $element = [
        '#type' => 'details',
        '#title' => $definition['label'],
        '#group' => 'toolbar][conditions][enabled][_tabs',
      ];

      // Add the plugin form as a subform.
      $plugin_form = [
        '#parents' => ['toolbar', 'conditions', 'enabled', $plugin_id],
      ];

      $plugin_form_state = SubformState::createForSubform($plugin_form, $form, $form_state);
      $plugin_form = $plugin->buildConfigurationForm($plugin_form, $plugin_form_state);

      $element['subform'] = $plugin_form;

      // Add the delete button.
      $element['actions'] = [
        '#type' => 'actions',
      ];

      $element['actions']['delete'] = [
        '#type' => 'submit',
        '#name' => 'delete_toolbar_condition_' . $plugin_id,
        '#value' => $this->t('Delete condition'),
        '#submit' => ['::deleteToolbarConditionSubmit'],
        '#limit_validation_errors' => [],
        '#ajax' => [
          'wrapper' => 'oidc-mcpf-toolbar-conditions',
          'callback' => '::toolbarConditionsAjax',
        ],
        '#plugin_id' => $plugin_id,
      ];

      $form['toolbar']['wrapper']['conditions']['enabled'][$plugin_id] = $element;
    }

    if (empty($options)) {
      return parent::buildForm($form, $form_state);
    }

    asort($options);

    $form['toolbar']['wrapper']['conditions']['add'] = [
      '#type' => 'details',
      '#title' => $this->t('Add visibility condition'),
      '#open' => TRUE,
      '#attributes' => [
        'class' => ['container-inline'],
      ],
    ];

    $form['toolbar']['wrapper']['conditions']['add']['plugin_id'] = [
      '#type' => 'select',
      '#title' => $this->t('Condition'),
      '#title_display' => 'invisible',
      '#options' => $options,
      '#empty_option' => '- ' . $this->t('Select') . ' -',
    ];

    $form['toolbar']['wrapper']['conditions']['add']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add'),
      '#validate' => ['::addToolbarConditionValidate'],
      '#submit' => ['::addToolbarConditionSubmit'],
      '#limit_validation_errors' => [
        ['toolbar', 'conditions', 'add'],
      ],
      '#ajax' => [
        'wrapper' => 'oidc-mcpf-toolbar-conditions',
        'callback' => '::toolbarConditionsAjax',
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    if (empty($form_state->getValue(['toolbar', 'enabled']))) {
      return;
    }

    // Ensure a widget ID is specified.
    if (empty($form_state->getValue(['toolbar', 'widget_id']))) {
      $form_state->setErrorByName('toolbar][widget_id', $this->t('@title field is required.', [
        '@title' => $this->t('Widget ID'),
      ]));
    }

    // Validate the conditions.
    foreach ($form_state->get('toolbar_conditions') as $plugin_id => $configuration) {
      $plugin_form = &$form['toolbar']['wrapper']['conditions']['enabled'][$plugin_id]['subform'];
      $plugin_form_state = SubformState::createForSubform($plugin_form, $form, $form_state);

      $plugin = $this->getConditionPlugin($plugin_id);
      $plugin->validateConfigurationForm($plugin_form, $plugin_form_state);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->config('oidc_mcpf.settings');

    // Get the toolbar configuration.
    $enabled_old = $config->get('toolbar.enabled');
    $enabled_new = (bool) $form_state->getValue(['toolbar', 'enabled']);
    $widget_id = NULL;
    $conditions = [];

    if ($enabled_new) {
      $widget_id = $form_state->getValue(['toolbar', 'widget_id']);

      foreach ($form_state->get('toolbar_conditions') as $plugin_id => $configuration) {
        $plugin_form = &$form['toolbar']['wrapper']['conditions']['enabled'][$plugin_id]['subform'];
        $plugin_form_state = SubformState::createForSubform($plugin_form, $form, $form_state);

        $plugin = $this->getConditionPlugin($plugin_id);
        $plugin->submitConfigurationForm($plugin_form, $plugin_form_state);

        $configuration = $plugin->getConfiguration();
        unset($configuration['id']);

        $conditions[$plugin_id] = $configuration;
      }
    }

    // Update the configuration.
    $config
      ->set('environment', $form_state->getValue('environment'))
      ->set('toolbar.enabled', $enabled_new)
      ->set('toolbar.widget_id', $widget_id)
      ->set('toolbar.language_switcher', (bool) $form_state->getValue(['toolbar', 'language_switcher']))
      ->set('toolbar.conditions', $conditions)
      ->save();

    if ($enabled_old || $enabled_new) {
      $this->libraryDiscovery->clearCachedDefinitions();
    }

    parent::submitForm($form, $form_state);
  }

  /**
   * Form validation handler; Validates the new toolbar condition.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function addToolbarConditionValidate(array &$form, FormStateInterface $form_state): void {
    if (empty($form_state->getValue(['toolbar', 'conditions', 'add', 'plugin_id']))) {
      $form_state->setErrorByName('toolbar][conditions][add][plugin_id', $this->t('Select a condition to add.'));
    }
  }

  /**
   * Form submit handler; Add a toolbar condition.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function addToolbarConditionSubmit(array &$form, FormStateInterface $form_state): void {
    $plugin_id = $form_state->getValue(['toolbar', 'conditions', 'add', 'plugin_id']);

    $conditions = &$form_state->get('toolbar_conditions');
    $conditions[$plugin_id] = [];

    $user_input = &$form_state->getUserInput();
    unset($user_input['toolbar']['conditions']['add']);

    $form_state->setRebuild();
  }

  /**
   * Form submit handler; Delete a toolbar condition.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function deleteToolbarConditionSubmit(array &$form, FormStateInterface $form_state): void {
    $plugin_id = $form_state->getTriggeringElement()['#plugin_id'];

    $visibility = &$form_state->get('toolbar_conditions');
    unset($visibility[$plugin_id]);

    $form_state->setRebuild();
  }

  /**
   * Ajax callback; Returns the ajax response the toolbar conditions changed.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array
   *   Replacement content for the IDM roles mapping element.
   */
  public function toolbarConditionsAjax(array $form, FormStateInterface $form_state): array {
    return $form['toolbar']['wrapper']['conditions'];
  }

  /**
   * Get a condition plugin instance.
   *
   * @param string $plugin_id
   *   The condition plugin ID.
   *
   * @return \Drupal\Core\Condition\ConditionInterface
   *   The condition plugin instance.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  protected function getConditionPlugin(string $plugin_id): ConditionInterface {
    if (!isset($this->conditionPlugins[$plugin_id])) {
      $this->conditionPlugins[$plugin_id] = $this->conditionManager->createInstance($plugin_id);
    }

    return $this->conditionPlugins[$plugin_id];
  }

}
