<?php

namespace Drupal\oidc_mcpf_user_purge;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\user\UserStorageInterface;

/**
 * The user purger service.
 */
class UserPurger implements UserPurgerInterface {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $database;

  /**
   * The user storage.
   *
   * @var \Drupal\user\UserStorageInterface
   */
  protected UserStorageInterface $userStorage;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected TimeInterface $time;

  /**
   * Class constructor.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function __construct(Connection $database, EntityTypeManagerInterface $entity_type_manager, TimeInterface $time) {
    $this->database = $database;
    $this->userStorage = $entity_type_manager->getStorage('user');
    $this->time = $time;
  }

  /**
   * {@inheritdoc}
   */
  public function purgeInactive(int $timeout, array $excluded_roles = [], int $limit = 50): void {
    if ($timeout < 3600) {
      throw new \InvalidArgumentException('$timeout must be at least 3600 seconds');
    }

    // Select up to $limit users that an ACM entry in the authmap
    // and are inactive to $timeout seconds.
    $access = $this->time->getCurrentTime() - $timeout;

    $query = $this->database->select('authmap', 'a')
      ->fields('a', ['uid'])
      ->condition('a.provider', 'oidc:acm')
      ->condition('ufd.access', $access, '<')
      ->orderBy('ufd.access')
      ->range(0, $limit);

    $query->innerJoin('users_field_data', 'ufd', 'ufd.uid = a.uid');

    // Add joins to exclude the specified roles.
    if (!empty($excluded_roles)) {
      $excluded_roles = array_values($excluded_roles);
      foreach ($excluded_roles as $index => $rid) {
        $query->leftJoin('user__roles', "ur$index", "a.uid = ur$index.entity_id AND ur$index.roles_target_id = :rid", [
          ':rid' => $rid,
        ]);
        $query->isNull("ur$index.entity_id");
      }
    }

    // Get the user IDs.
    $uids = $query->execute()->fetchCol();

    if (empty($uids)) {
      return;
    }

    // Load the users and delete them.
    $users = $this->userStorage->loadMultiple($uids);
    $this->userStorage->delete($users);
  }

}
