<?php

namespace Drupal\oidc_mcpf\Controller;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Session\AnonymousUserSession;
use Drupal\Core\Session\SessionManagerInterface;
use Drupal\Core\Url;
use Drupal\oidc\OpenidConnectSessionInterface;
use Drupal\oidc\Routing\ImmutableTrustedRedirectResponse;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Contrller for the ACM OpenID Connect realm specific routes.
 */
class AcmController extends ControllerBase {

  /**
   * Class constructor.
   *
   * @param \Drupal\oidc\OpenidConnectSessionInterface $session
   *   The OpenID Connect session.
   * @param \Drupal\Core\Session\SessionManagerInterface $sessionManager
   *   The session manager.
   */
  public function __construct(
    protected OpenidConnectSessionInterface $session,
    protected SessionManagerInterface $sessionManager,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('oidc.openid_connect_session'),
      $container->get('session_manager')
    );
  }

  /**
   * Switch ACM capacity.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   *
   * @return \Drupal\oidc\Routing\ImmutableTrustedRedirectResponse
   *   The immutable redirect response.
   */
  public function switchCapacity(Request $request): ImmutableTrustedRedirectResponse {
    // Log out of Drupal. This code is copied from user_logout() because that function
    // also destroys the PHP session, which prevents us from starting a new session.
    $user = $this->currentUser();
    $this->getLogger('user')->notice('Session closed for %name.', ['%name' => $user->getAccountName()]);
    $this->moduleHandler()->invokeAll('user_logout', [$user]);
    $user->setAccount(new AnonymousUserSession());

    // Clear the session and restore the realm.
    $this->sessionManager->clear();
    $this->session->initRealm('acm');

    // Initialize the state.
    $destination = $request->query->get('destination');
    $state = $this->session->initState($destination);

    // Get the login URL.
    $url = $this->session->getRealmPlugin()
      ->getLoginUrl($state, Url::fromRoute('oidc.openid_connect.login_redirect'))
      ->setAbsolute();

    // Add the login hint.
    /** @var array $query */
    $query = $url->getOption('query');
    $query['login_hint'] = 'eyJzd2l0Y2hfaWQiOiB0cnVlfQ==';
    $url->setOption('query', $query);

    // Redirect.
    return new ImmutableTrustedRedirectResponse($url);
  }

  /**
   * Check if the user may access the switch capacity callback.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function switchCapacityAccess(): AccessResultInterface {
    $audiences = $this->config('oidc.realm.acm')->get('audiences') ?? [];

    return AccessResult::allowedIf(
      count($audiences) > 1 &&
      $this->session->isAuthenticated() &&
      $this->session->getRealmPluginId() === 'acm'
    )->addCacheTags(['config:oidc.realm.acm']);
  }

}
