<?php

namespace Drupal\oidc_mcpf\EventSubscriber;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\oidc\JsonWebTokens;
use Drupal\oidc\OpenidConnectSessionInterface;
use Drupal\externalauth\Event\ExternalAuthEvents;
use Drupal\externalauth\Event\ExternalAuthLoginEvent;
use Drupal\oidc_mcpf\Audience;
use Drupal\oidc_mcpf\Plugin\OpenidConnectRealm\AcmOpenidConnectRealm;
use Drupal\oidc_mcpf\RoleMapType;
use Drupal\user\UserInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Event subscriber to update the synced user fields and roles on login.
 */
class UpdateUserSubscriber implements EventSubscriberInterface {

  /**
   * The OpenID Connect session service.
   *
   * @var \Drupal\oidc\OpenidConnectSessionInterface
   */
  protected OpenidConnectSessionInterface $session;

  /**
   * The user storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected EntityStorageInterface $userStorage;

  /**
   * Class constructor.
   *
   * @param \Drupal\oidc\OpenidConnectSessionInterface $session
   *   The OpenID Connect session service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function __construct(OpenidConnectSessionInterface $session, EntityTypeManagerInterface $entity_type_manager) {
    $this->session = $session;
    $this->userStorage = $entity_type_manager->getStorage('user');
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    $events[ExternalAuthEvents::LOGIN][] = 'onLogin';

    return $events;
  }

  /**
   * Updates the synced user fields and roles on login.
   *
   * @param \Drupal\externalauth\Event\ExternalAuthLoginEvent $event
   *   The login event.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function onLogin(ExternalAuthLoginEvent $event): void {
    $plugin = $this->session->getRealmPlugin();

    if (!$plugin instanceof AcmOpenidConnectRealm) {
      return;
    }

    $tokens = $this->session->getJsonWebTokens();
    $audience = $tokens->getClaim('vo_doelgroepcode') ?: Audience::CITIZEN;
    $account = $event->getAccount();
    $save = FALSE;

    // Update the fields if not a citizen.
    if ($audience !== Audience::CITIZEN) {
      $save = $this->syncUserFields($account, $tokens);
    }

    // Sync the roles.
    $configuration = $plugin->getConfiguration();

    if (!empty($configuration['roles_mapping'])) {
      $idm_roles = [];
      if (!empty($configuration['idm_claim'])) {
        $idm_roles = $tokens->getClaim($configuration['idm_claim']) ?: [];

        if (is_string($idm_roles)) {
          $idm_roles = explode(',', $idm_roles);
        }
      }

      $save |= $this->syncUserRoles($account, $configuration['roles_mapping'], $audience, $idm_roles);
    }

    // Save if there are changes.
    if ($save) {
      $account->save();
    }
  }

  /**
   * Synchronize our user fields with their matching claims.
   *
   * @param \Drupal\user\UserInterface $account
   *   The user that just logged in.
   * @param \Drupal\oidc\JsonWebTokens $tokens
   *   The received JSON web tokens.
   *
   * @return bool
   *   Boolean indicating if the account was updated.
   */
  protected function syncUserFields(UserInterface $account, JsonWebTokens $tokens): bool {
    $updated = FALSE;

    // Update the phone number.
    $phone = $tokens->getClaim('phone_number');

    if ($phone !== NULL && $phone !== $account->get('phone')->value) {
      $account->set('phone', $phone);
      $updated = TRUE;
    }

    // Update the organization name.
    $organization_name = $tokens->getClaim('vo_orgnaam');

    if ($organization_name !== NULL && $organization_name !== $account->get('organization_name')->value) {
      $account->set('organization_name', $organization_name);
      $updated = TRUE;
    }

    // Update the organization code.
    $organization_code = $tokens->getClaim('vo_orgcode');

    if ($organization_code !== NULL && $organization_code !== $account->get('organization_code')->value) {
      $account->set('organization_code', $organization_code);
      $updated = TRUE;
    }

    return $updated;
  }

  /**
   * Synchronize the Drupal roles with their matching IDM roles.
   *
   * @param \Drupal\user\UserInterface $account
   *   The user that just logged in.
   * @param array $roles_mapping
   *   The roles mapping.
   * @param array $audience
   *   The audience.
   * @param array $idm_roles
   *   The retrieved IDM roles.
   *
   * @return bool
   *   Boolean indicating if roles were changed.
   */
  protected function syncUserRoles(UserInterface $account, array $roles_mapping, string $audience, array $idm_roles): bool {
    $updated = FALSE;

    // Duplicate all 3D roles (<role>-<context>:<scope>) as 2D (<role>:<context>) roles.
    if (!empty($idm_roles) && str_contains(implode('', $idm_roles), '-')) {
      $tmp = $idm_roles;
      foreach ($tmp as $idm_role) {
        if (str_contains($idm_role, '-')) {
          $idm_roles[] = str_replace('-', ':', strstr($idm_role, ':', TRUE));
        }
      }
    }

    // Determine which roles should be added and removed.
    $roles_add = [];
    $roles_remove = [];

    foreach ($roles_mapping as $entry) {
      switch ($entry['type']) {
        case RoleMapType::AUDIENCE:
          if ($entry['value'] === $audience) {
            $roles_add[] = $entry['rid'];
          }
          else {
            $roles_remove[] = $entry['rid'];
          }
          break;

        case RoleMapType::IDM:
          if (in_array($entry['value'], $idm_roles, TRUE)) {
            $roles_add[] = $entry['rid'];
          }
          else {
            $roles_remove[] = $entry['rid'];
          }
          break;
      }
    }

    // Get the existing roles.
    $account_roles = $account->getRoles(TRUE);

    // Add new roles.
    if (!empty($roles_add)) {
      $roles_add = array_diff($roles_add, $account_roles);

      if (!empty($roles_add)) {
        foreach ($roles_add as $rid) {
          $account->addRole($rid);
        }

        $updated = TRUE;
      }
    }

    // Remove old roles.
    if (!empty($roles_remove)) {
      $roles_remove = array_intersect($account_roles, $roles_remove);

      if (!empty($roles_remove)) {
        foreach ($roles_remove as $rid) {
          $account->removeRole($rid);
        }

        $updated = TRUE;
      }
    }

    return $updated;
  }

}
