<?php

namespace Drupal\oidc_mcpf\Plugin\OpenidConnectRealm;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Component\Render\MarkupInterface;
use Drupal\Component\Utility\Html;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\KeyValueStore\KeyValueFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Render\Markup;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Url;
use Drupal\oidc\DisplayNameFormatOptionsTrait;
use Drupal\oidc\JsonHttp\JsonHttpClientInterface;
use Drupal\oidc\JsonHttp\JsonHttpPostRequestInterface;
use Drupal\oidc\JsonWebTokens;
use Drupal\oidc\OpenidConnectLoginException;
use Drupal\oidc\OpenidConnectRealm\OpenidConnectRealmBase;
use Drupal\oidc\OpenidConnectRealm\OpenidConnectRealmConfigurableInterface;
use Drupal\oidc_mcpf\Audience;
use Drupal\oidc_mcpf\RoleMapType;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides the ACM OpenID Connect realm plugin.
 *
 * @OpenidConnectRealm(
 *   id = "acm",
 *   name = @Translation("ACM"),
 * )
 */
class AcmOpenidConnectRealm extends OpenidConnectRealmBase implements OpenidConnectRealmConfigurableInterface {

  use DisplayNameFormatOptionsTrait;

  /**
   * The audience constants.
   *
   * @deprecated Use the constants from \Drupal\oidc_mcpf\Audience
   */
  public const AUDIENCE_CITIZEN = Audience::CITIZEN;
  public const AUDIENCE_ORGANIZATION = Audience::ASSOCIATION;
  public const AUDIENCE_ASSOCIATION = Audience::ASSOCIATION;
  public const AUDIENCE_EDUCATION = Audience::EDUCATION;
  public const AUDIENCE_GOV_FLEMISH = Audience::GOV_FLEMISH;
  public const AUDIENCE_GOV_LOCAL = Audience::GOV_LOCAL;

  /**
   * List of scopes that are automatically added.
   */
  protected const FIXED_SCOPES = ['openid', 'profile', 'phone', 'vo'];

  /**
   * The environment.
   *
   * @var string
   */
  protected string $environment;

  /**
   * The user role storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected EntityStorageInterface $userRoleStorage;

  /**
   * Class constructor.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param array $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\KeyValueStore\KeyValueFactoryInterface $key_value_factory
   *   The key-value storage factory.
   * @param \Drupal\oidc\JsonHttp\JsonHttpClientInterface $json_http_client
   *   The JSON HTTP client.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger_channel
   *   The logger channel.
   */
  public function __construct(array $configuration, string $plugin_id, array $plugin_definition, ConfigFactoryInterface $config_factory, EntityTypeManagerInterface $entity_type_manager, KeyValueFactoryInterface $key_value_factory, JsonHttpClientInterface $json_http_client, TimeInterface $time, LoggerChannelInterface $logger_channel) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $key_value_factory, $json_http_client, $time, $logger_channel);

    $this->setConfiguration($configuration);
    $this->environment = $config_factory->get('oidc_mcpf.settings')->get('environment');
    $this->userRoleStorage = $entity_type_manager->getStorage('user_role');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('entity_type.manager'),
      $container->get('keyvalue'),
      $container->get('oidc.json_http_client'),
      $container->get('datetime.time'),
      $container->get('logger.channel.oidc')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getConfiguration(): array {
    return $this->configuration;
  }

  /**
   * {@inheritdoc}
   */
  public function setConfiguration(array $configuration): void {
    $this->configuration = array_merge(
      $this->defaultConfiguration(),
      $configuration
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'enabled' => FALSE,
      'client_id' => NULL,
      'client_secret' => NULL,
      'audiences' => [Audience::CITIZEN],
      'scopes' => [],
      'display_name_format' => '[user:name]',
      'idm_claim' => NULL,
      'roles_mapping' => [],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enabled'),
      '#default_value' => (int) $this->configuration['enabled'],
    ];

    $form['wrapper'] = [
      '#type' => 'container',
      '#parents' => ['realms', 'acm', 'settings'],
      '#states' => [
        'visible' => [
          ':checkbox[name="realms[acm][settings][enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['wrapper']['environment'] = [
      '#type' => 'markup',
      '#markup' => $this->t('Note that the environment is <a href=":url">configured in the settings</a>.', [
        ':url' => Url::fromRoute('oidc_mcpf.admin.settings')->toString(),
      ]),
    ];

    $form['wrapper']['client_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Client ID'),
      '#default_value' => $this->configuration['client_id'],
      '#states' => [
        'required' => [
          ':checkbox[name="realms[acm][settings][enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['wrapper']['client_secret'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Client secret'),
      '#default_value' => $this->configuration['client_secret'],
      '#states' => [
        'required' => [
          ':checkbox[name="realms[acm][settings][enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['wrapper']['audiences'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Audiences'),
      '#description' => $this->t('Note that the chosen audiences must be enabled on ACM as well. Additionally the %audience and %code claims must be included in the response.', [
        '%audience' => 'vo_doelgroepcode',
        '%code' => 'vo_orcode',
      ]),
      '#options' => Audience::getOptions(),
      '#default_value' => $this->configuration['audiences'],
      '#required' => TRUE,
    ];

    $form['wrapper']['scopes'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Additional scopes'),
      '#description' => $this->t('Following scopes will be added automatically: @scopes', [
        '@scopes' => $this->formatList(self::FIXED_SCOPES, TRUE),
      ]),
      '#default_value' => implode(' ', $this->configuration['scopes']),
    ];

    $form['wrapper']['display_name_format'] = [
      '#type' => 'select',
      '#title' => $this->t('Display name format'),
      '#options' => $this->buildDisplayNameFormatOptions([
        '[user:account-name]',
        '[user:given-name]',
        '[user:given-name] [user:family-name]',
        '[user:given-name] [user:family-name-abbr]',
        '[user:mail]',
        '[user:organization-name]',
        '[user:organization-code]',
        '[user:organization-name] ([user:organization-code])',
        '[user:given-name] ([user:organization-name])',
        '[user:given-name] [user:family-name] ([user:organization-name])',
        '[user:given-name] [user:family-name-abbr] ([user:organization-name])',
        '[user:given-name] ([user:organization-code])',
        '[user:given-name] [user:family-name] ([user:organization-code])',
        '[user:given-name] [user:family-name-abbr] ([user:organization-code])',
      ]),
      '#default_value' => $this->configuration['display_name_format'],
      '#required' => TRUE,
    ];

    $form['wrapper']['idm_claim'] = [
      '#type' => 'textfield',
      '#title' => $this->t('IDM claim'),
      '#description' => $this->t("Don't forget to specify the scope that provides it as well."),
      '#default_value' => $this->configuration['idm_claim'],
    ];

    // Get the user roles.
    $roles = $this->userRoleStorage->loadMultiple();
    unset($roles[AccountInterface::ANONYMOUS_ROLE], $roles[AccountInterface::AUTHENTICATED_ROLE]);

    // Turn into an [ID => label] array.
    array_walk($roles, function (&$role): void {
      $role = $role->label();
    });

    // Store the roles mapping.
    if (!$form_state->has('roles_mapping')) {
      $roles_mapping = $this->configuration['roles_mapping'];
      $roles_mapping = array_filter($roles_mapping, function ($entry) use ($roles): bool {
        return isset($roles[$entry['rid']]);
      });

      $form_state->set('roles_mapping', $roles_mapping);
    }

    $form['wrapper']['roles_mapping'] = [
      '#type' => 'details',
      '#title' => $this->t('Role assignment'),
      '#description' => $this->t('Assign a Drupal roles to a specific audiences or IDM roles. Note that you must specify the IDM role in the same format as configured in ACM: @formats. In case ACM provides the roles with a scope, the format with role and context is also matched.', [
        '@formats' => $this->formatList([
          '<role>-<context>:<scope>',
          '<role>:<context>',
          '<role>',
        ]),
      ]),
      '#open' => TRUE,
      '#prefix' => '<div id="oidc-mcpf-roles-mapping">',
      '#suffix' => '</div>',
    ];

    $form['wrapper']['roles_mapping']['table'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Drupal role'),
        $this->t('Type'),
        $this->t('Audience or IDM role'),
        $this->t('Delete'),
      ],
      '#empty' => empty($roles) ? $this->t('There are no Drupal roles that can be assigned.') : $this->t('No roles to assign have been specified.'),
    ];

    foreach ($form_state->get('roles_mapping') as $key => $entry) {
      $row = [];

      $row['rid']['value'] = [
        '#type' => 'value',
        '#value' => $entry['rid'],
        '#parents' => ['realms', 'acm', 'settings', 'roles_mapping', 'table', $key, 'rid'],
      ];

      $row['rid']['label'] = [
        '#plain_text' => $roles[$entry['rid']],
      ];

      $row['type']['select'] = [
        '#type' => 'select',
        '#title' => $this->t('Type'),
        '#title_display' => 'invisible',
        '#options' => RoleMapType::getOptions(),
        '#default_value' => $entry['type'] ?? RoleMapType::AUDIENCE,
        '#required' => TRUE,
        '#parents' => ['realms', 'acm', 'settings', 'roles_mapping', 'table', $key, 'type'],
        '#ajax' => [
          'wrapper' => 'oidc-mcpf-roles-mapping-' . $key . '-value',
          'callback' => '',
          'trigger_as' => [
            'name' => 'set_role_mapping_type_' . $key,
            'value' => $this->t('Set type'),
          ],
        ],
      ];

      $row['type']['submit'] = [
        '#type' => 'submit',
        '#name' => 'set_role_mapping_type_' . $key,
        '#value' => $this->t('Set type'),
        '#submit' => [
          [static::class, 'setRolesMappingEntryTypeSubmit'],
        ],
        '#limit_validation_errors' => [
          ['realms', 'acm', 'settings', 'roles_mapping', 'table', $key, 'type'],
        ],
        '#ajax' => [
          'wrapper' => '',
          'callback' => [static::class, 'setRolesMappingEntryTypeAjax'],
        ],
        '#parents' => ['realms', 'acm', 'settings', 'roles_mapping', 'table', $key, 'type_submit'],
        '#attributes' => [
          'class' => ['js-hide'],
        ],
        '#mapping_key' => $key,
      ];

      $row['value'] = [
        '#title_display' => 'invisible',
        '#default_value' => $entry['value'] ?? NULL,
        '#required' => TRUE,
        '#prefix' => '<div id="oidc-mcpf-roles-mapping-' . $key . '-value">',
        '#suffix' => '</div>',
      ];

      if ($entry['type'] === RoleMapType::IDM) {
        $row['value'] += [
          '#type' => 'textfield',
          '#title' => $this->t('IDM role'),
        ];

      }
      else {
        $row['value'] += [
          '#type' => 'select',
          '#title' => $this->t('Audience'),
          '#options' => Audience::getOptions(),
          '#empty_option' => '- ' . $this->t('Select') . ' -',
        ];
      }

      $row['delete'] = [
        '#type' => 'submit',
        '#name' => 'delete_role_mapping_' . $key,
        '#value' => $this->t('Delete'),
        '#submit' => [
          [static::class, 'deleteRolesMappingEntrySubmit'],
        ],
        '#limit_validation_errors' => [],
        '#ajax' => [
          'wrapper' => 'oidc-mcpf-roles-mapping',
          'callback' => [static::class, 'rolesMappingAjax'],
        ],
        '#mapping_key' => $key,
      ];

      $form['wrapper']['roles_mapping']['table'][$key] = $row;

      unset($roles[$entry['rid']]);
    }

    if (empty($roles)) {
      return $form;
    }

    $form['wrapper']['roles_mapping']['add'] = [
      '#type' => 'details',
      '#title' => $this->t('Add role'),
      '#open' => TRUE,
      '#attributes' => [
        'class' => ['container-inline'],
      ],
    ];

    $form['wrapper']['roles_mapping']['add']['rid'] = [
      '#type' => 'select',
      '#title' => $this->t('Drupal role'),
      '#title_display' => 'invisible',
      '#options' => $roles,
      '#empty_option' => '- ' . $this->t('Select') . ' -',
    ];

    $form['wrapper']['roles_mapping']['add']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add'),
      '#validate' => [
        [static::class, 'addRolesMappingEntryValidate'],
      ],
      '#submit' => [
        [static::class, 'addRolesMappingEntrySubmit'],
      ],
      '#limit_validation_errors' => [
        ['realms', 'acm', 'settings', 'roles_mapping', 'add'],
      ],
      '#ajax' => [
        'wrapper' => 'oidc-mcpf-roles-mapping',
        'callback' => [static::class, 'rolesMappingAjax'],
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $triggering_element = $form_state->getTriggeringElement();

    if (empty($form_state->getValue('enabled')) || isset($triggering_element['#ajax'])) {
      return;
    }

    // Ensure a client ID is specified.
    if (empty($form_state->getValue('client_id'))) {
      $form_state->setErrorByName('client_id', $this->t('@title field is required.', [
        '@title' => $this->t('Client ID'),
      ]));
    }

    // Ensure a client secret is specified.
    if (empty($form_state->getValue('client_secret'))) {
      $form_state->setErrorByName('client_secret', $this->t('@title field is required.', [
        '@title' => $this->t('Client secret'),
      ]));
    }

    // Check if the audiences and display name format are compatible.
    $display_name_format = $form_state->getValue('display_name_format');

    if (str_starts_with($display_name_format, '[user:organization-')) {
      $audiences = $form_state->getValue('audiences');
      $audiences = array_values(array_diff($audiences, ['0']));

      if (in_array(Audience::CITIZEN, $audiences, TRUE)) {
        if (count($audiences) === 1) {
          $form_state->setErrorByName('display_name_format', $this->t('The display name format cannot contain organization details if they are not allow to log in.'));
        }
        else {
          $display_name_format = preg_replace('/\[user:organization-[a-z]+\]/', '', $display_name_format);

          if (!str_contains($display_name_format, '[user:')) {
            $form_state->setErrorByName('display_name_format', $this->t('The display name format may not contain only organization details if citizens are allowed to log in.'));
          }
        }
      }
    }

    // Validate the roles mapping.
    $roles_mapping = $form_state->getValue(['roles_mapping', 'table']);

    if (empty($roles_mapping)) {
      return;
    }

    $has_idm_claim = !empty($form_state->getValue('idm_claim'));

    foreach ($roles_mapping as $key => $entry) {
      if ($entry['type'] !== RoleMapType::IDM) {
        continue;
      }

      if (!$has_idm_claim) {
        $form_state->setErrorByName(
          "roles_mapping][table][$key][type",
          $this->t("It's not possible to assign a role based on IDM because no IDM claim has been specified.")
        );
      }
      elseif (!preg_match('/^[A-Za-z]+(?::[a-z]+|-[a-z0-9]+:[A-Z0-9]+)?$/', $entry['value'])) {
        $form_state->setErrorByName("roles_mapping][table][$key][value", $this->t('%value is not a valid IDM role.', [
          '%value' => $entry['value'],
        ]));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $audiences = $form_state->getValue('audiences');
    $audiences = array_values(array_diff($audiences, ['0']));

    $scopes = $form_state->getValue('scopes');

    if (!empty($scopes)) {
      $scopes = preg_split('/\s+/', str_replace(',', ' ', $scopes));
      $scopes = array_diff($scopes, self::FIXED_SCOPES);
      $scopes = array_unique($scopes);
    }

    $roles_mapping = $form_state->getValue(['roles_mapping', 'table']);
    array_walk($roles_mapping, function (array &$entry) {
      $entry = array_intersect_key($entry, [
        'rid' => '',
        'type' => '',
        'value' => '',
      ]);
    });

    $this->configuration['enabled'] = (bool) $form_state->getValue('enabled');
    $this->configuration['client_id'] = $form_state->getValue('client_id');
    $this->configuration['client_secret'] = $form_state->getValue('client_secret');
    $this->configuration['audiences'] = $audiences;
    $this->configuration['scopes'] = $scopes ?: [];
    $this->configuration['display_name_format'] = $form_state->getValue('display_name_format');
    $this->configuration['idm_claim'] = $form_state->getValue('idm_claim') ?: NULL;
    $this->configuration['roles_mapping'] = $roles_mapping;
  }

  /**
   * Form validation handler; Validates the new roles mapping entry.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public static function addRolesMappingEntryValidate(array &$form, FormStateInterface $form_state): void {
    if (empty($form_state->getValue(['realms', 'acm', 'settings', 'roles_mapping', 'add', 'rid']))) {
      $form_state->setErrorByName('realms][acm][settings][roles_mapping][add][rid', t('Select a role to add.'));
    }
  }

  /**
   * Form submit handler; Add an entry to the IDM roles mapping.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public static function addRolesMappingEntrySubmit(array &$form, FormStateInterface $form_state): void {
    $rid = $form_state->getValue(['realms', 'acm', 'settings', 'roles_mapping', 'add', 'rid']);

    $roles_mapping = &$form_state->get('roles_mapping');
    $roles_mapping[] = [
      'rid' => $rid,
      'type' => RoleMapType::AUDIENCE,
    ];

    $user_input = &$form_state->getUserInput();
    unset($user_input['realms']['acm']['settings']['roles_mapping']['add']);

    $form_state->setRebuild();
  }

  /**
   * Form submit handler; Delete an entry from the roles mapping.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public static function deleteRolesMappingEntrySubmit(array &$form, FormStateInterface $form_state): void {
    $key = $form_state->getTriggeringElement()['#mapping_key'];

    $roles_mapping = &$form_state->get('roles_mapping');
    unset($roles_mapping[$key]);

    $form_state->setRebuild();
  }

  /**
   * Ajax callback; Returns the ajax response when the roles mapping was changed.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array
   *   Replacement content for the roles mapping element.
   */
  public static function rolesMappingAjax(array $form, FormStateInterface $form_state): array {
    $triggering_element = $form_state->getTriggeringElement();
    $array_parents = array_slice(
      $triggering_element['#array_parents'],
      0,
      isset($triggering_element['#mapping_key']) ? -3 : -2
    );

    return NestedArray::getValue($form, $array_parents);
  }

  /**
   * Form submit handler; Set the type of a roles mapping entry.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public static function setRolesMappingEntryTypeSubmit(array &$form, FormStateInterface $form_state): void {
    $key = $form_state->getTriggeringElement()['#mapping_key'];
    $type = $form_state->getValue(['realms', 'acm', 'settings', 'roles_mapping', 'table', $key, 'type']);

    $roles_mapping = &$form_state->get('roles_mapping');
    $roles_mapping[$key]['type'] = $type;
    unset($roles_mapping[$key]['value']);

    $user_input = &$form_state->getUserInput();
    unset($user_input['realms']['acm']['settings']['roles_mapping']['table'][$key]);

    $form_state->setRebuild();
  }

  /**
   * Ajax callback; Returns the ajax response when setting the roles mapping entry type.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array
   *   Replacement content for the value element of the roles mapping entry.
   */
  public static function setRolesMappingEntryTypeAjax(array $form, FormStateInterface $form_state): array {
    $array_parents = $form_state->getTriggeringElement()['#array_parents'];
    $array_parents = array_slice($array_parents, 0, -2);
    $array_parents[] = 'value';

    return NestedArray::getValue($form, $array_parents);
  }

  /**
   * {@inheritdoc}
   */
  public function isEnabled(): bool {
    return $this->configuration['enabled'];
  }

  /**
   * {@inheritdoc}
   */
  public function getJsonWebTokensForLogin($state, $code): JsonWebTokens {
    $tokens = parent::getJsonWebTokensForLogin($state, $code);
    $audience = $tokens->getClaim('vo_doelgroepcode') ?? Audience::CITIZEN;

    if (in_array($audience, $this->configuration['audiences'], TRUE)) {
      // An empty ID claim means the vo_orgcode claim is missing.
      if ($tokens->getId() === '') {
        $message = $this->t('The %code claim must be present to login as an organization.', [
          '%code' => 'vo_orgcode',
        ]);

        throw new OpenidConnectLoginException($message);
      }

      return $tokens;
    }

    // Audience not allowed.
    if ($audience === Audience::CITIZEN) {
      $message = $this->t('You are not allowed to login as a citizen, please select an organization to represent.');
    }
    else {
      $message = $this->t('You are not allowed to login as a representative of the chosen organization.');
    }

    throw new OpenidConnectLoginException($message);
  }

  /**
   * {@inheritdoc}
   */
  public function getDisplayNameFormat(): string {
    return $this->configuration['display_name_format'];
  }

  /**
   * {@inheritdoc}
   */
  protected function getClientId(): ?string {
    return $this->configuration['client_id'];
  }

  /**
   * {@inheritdoc}
   */
  protected function getClientSecret(): ?string {
    return $this->configuration['client_secret'];
  }

  /**
   * {@inheritdoc}
   */
  protected function getScopes(): array {
    $scopes = $this->configuration['scopes'];
    $scopes[] = 'profile';

    if (!empty(array_diff($this->configuration['audiences'], [Audience::CITIZEN]))) {
      $scopes[] = 'phone';
      $scopes[] = 'vo';
    }

    return $scopes;
  }

  /**
   * {@inheritdoc}
   */
  protected function getIssuer(): string {
    $host = 'authenticatie.vlaanderen.be';

    if ($this->environment === 'test') {
      $host = 'authenticatie-ti.vlaanderen.be';
    }

    return 'https://' . $host . '/op';
  }

  /**
   * {@inheritdoc}
   */
  protected function getAuthorizationEndpoint(): string {
    return $this->getIssuer() . '/v1/auth';
  }

  /**
   * {@inheritdoc}
   */
  protected function getTokenEndpoint(): string {
    return $this->getIssuer() . '/v1/token';
  }

  /**
   * {@inheritdoc}
   */
  protected function getUserinfoEndpoint(): ?string {
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  protected function getEndSessionEndpoint(): string {
    return $this->getIssuer() . '/v1/logout';
  }

  /**
   * {@inheritdoc}
   */
  protected function getJwksUrl(): string {
    return $this->getIssuer() . '/v1/keys';
  }

  /**
   * {@inheritdoc}
   */
  protected function getJsonWebTokens(JsonHttpPostRequestInterface $json_http_post_request): JsonWebTokens {
    $tokens = parent::getJsonWebTokens($json_http_post_request);
    $audience = $tokens->getClaim('vo_doelgroepcode') ?? Audience::CITIZEN;

    if ($audience === Audience::CITIZEN) {
      return $tokens;
    }

    // Build the audience ID.
    $audience_id = (string) $tokens->getClaim('vo_orgcode');

    if ($audience_id !== '') {
      $audience_id = $audience . '_' . $audience_id . '_' . $tokens->getId();
    }

    // Update the claims.
    $tokens->setClaim('audience_id', $audience_id);
    $tokens->setIdClaim('audience_id');
    $tokens->setEmailClaim('vo_email');

    return $tokens;
  }

  /**
   * Format a list of strings.
   *
   * @param string[] $list
   *   The list of strings.
   * @param bool $and
   *   Set to TRUE to use "and" instead of "or" before the last item.
   *
   * @return \Drupal\Component\Render\MarkupInterface
   *   The formatted list.
   */
  protected function formatList(array $list, bool $and = FALSE): MarkupInterface {
    $last = array_pop($list);

    if (empty($list)) {
      $markup = Html::escape($last);
      $markup = '<em class="placeholder">' . $markup . '</em>';

      return Markup::create($markup);
    }

    $args = [
      '%value' => $last,
    ];

    if ($and) {
      $last = $this->t('and %value', $args);
    }
    else {
      $last = $this->t('or %value', $args);
    }

    array_walk($list, function (&$item) {
      $item = Html::escape($item);
      $item = '<em class="placeholder">' . $item . '</em>';
    });

    return Markup::create(implode(', ', $list) . ' ' . $last);
  }

}
