<?php

namespace Drupal\oidc_mcpf;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Condition\ConditionInterface;
use Drupal\Core\Condition\ConditionManager;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Path\PathMatcherInterface;
use Drupal\Core\Plugin\Context\ContextHandlerInterface;
use Drupal\Core\Plugin\Context\ContextRepositoryInterface;
use Drupal\Core\Plugin\ContextAwarePluginInterface;
use Drupal\Core\Routing\RedirectDestinationTrait;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Url;
use Drupal\oidc\OpenidConnectSessionInterface;

/**
 * The toolbar service.
 */
class Toolbar implements ToolbarInterface {

  use RedirectDestinationTrait;

  /**
   * Class constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\oidc\OpenidConnectSessionInterface $session
   *   The OpenID Connect session.
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   The current user.
   * @param \Drupal\Core\Condition\ConditionManager $conditionManager
   *   The condition plugin manager.
   * @param \Drupal\Core\Plugin\Context\ContextHandlerInterface $contextHandler
   *   The context handler.
   * @param \Drupal\Core\Plugin\Context\ContextRepositoryInterface $contextRepository
   *   The context repository.
   * @param \Drupal\Core\Language\LanguageManagerInterface $languageManager
   *   The language manager.
   * @param \Drupal\Core\Path\PathMatcherInterface $pathMatcher
   *   The path matcher.
   * @param \Drupal\Core\Routing\RouteMatchInterface $currentRouteMatch
   *   The current route match.
   */
  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected OpenidConnectSessionInterface $session,
    protected AccountInterface $currentUser,
    protected ConditionManager $conditionManager,
    protected ContextHandlerInterface $contextHandler,
    protected ContextRepositoryInterface $contextRepository,
    protected LanguageManagerInterface $languageManager,
    protected PathMatcherInterface $pathMatcher,
    protected RouteMatchInterface $currentRouteMatch,
  ) {

  }

  /**
   * {@inheritdoc}
   */
  public function alterLibrary(array &$library): void {
    $settings = $this->configFactory->get('oidc_mcpf.settings');
    $widget_id = $settings->get('toolbar.widget_id');

    if (empty($widget_id)) {
      return;
    }

    if ($settings->get('environment') === 'test') {
      $host = 'tni.widgets.burgerprofiel.dev-vlaanderen.be';
    }
    else {
      $host = 'prod.widgets.vlaanderen.be';
    }

    $library['js']['https://' . $host . '/api/v2/widget/' . $widget_id . '/embed'] = [
      'type' => 'external',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function access(): AccessResultInterface {
    // Neutral by default.
    $result = AccessResult::neutral()
      ->addCacheTags(['config:oidc_mcpf.settings']);

    // The toolbar must be enabled.
    $settings = $this->configFactory->get('oidc_mcpf.settings');

    if (!$settings->get('toolbar.enabled')) {
      return $result->andIf(AccessResult::forbidden('Toolbar not enabled'));
    }

    // Check the conditions.
    $conditions = $settings->get('toolbar.conditions') ?? [];

    foreach ($conditions as $plugin_id => $configuration) {
      if (!$this->conditionManager->hasDefinition($plugin_id)) {
        continue;
      }

      $condition = $this->getConfiguredCondition($plugin_id, $configuration);

      try {
        $result = $result->andIf(AccessResult::forbiddenIf(
          !$this->conditionManager->execute($condition),
          "Condition $plugin_id failed evaluation"
        ));
      }
      catch (\Throwable $ex) {
        $result = $result->andIf(AccessResult::forbidden($ex->getMessage()));
      }

      $result->addCacheableDependency($condition);

      if ($result->isForbidden()) {
        return $result;
      }
    }

    // Allow access if the user ain't authenticated.
    if (!$this->currentUser->isAuthenticated()) {
      return $result->andIf(AccessResult::allowed()->addCacheContexts(['user.roles:anonymous']));
    }

    // Differ on wether it's an OpenID session and the realm.
    $result->addCacheContexts([
      'user.openid_connect',
      'user.openid_connect_realm',
    ]);

    // Allow if authenticated via ACM.
    return $result->andIf(AccessResult::allowedIf($this->isAcmAuthenticated()));
  }

  /**
   * {@inheritdoc}
   */
  public function attach(array &$attachments): void {
    $metadata = new CacheableMetadata();
    $attachments['#attached']['library'][] = 'oidc_mcpf/toolbar';
    $attachments['#attached']['drupalSettings']['oidcMcpf']['toolbar'] = [
      'accessMenu' => $this->getAccessMenuSettings($metadata),
      'languageSwitcher' => $this->getLanguageSwitcherSettings($metadata),
    ];

    $metadata->applyTo($attachments);
  }

  /**
   * Get a configured condition instance.
   *
   * @param string $plugin_id
   *   The condition plugin ID.
   * @param array $configuration
   *   The plugin configuration.
   *
   * @return \Drupal\Core\Condition\ConditionInterface
   *   The configured plugin instance.
   */
  protected function getConfiguredCondition(string $plugin_id, array $configuration = []): ConditionInterface {
    /** @var \Drupal\Core\Condition\ConditionInterface $condition */
    $condition = $this->conditionManager->createInstance($plugin_id, $configuration);

    if (!$condition instanceof ContextAwarePluginInterface) {
      return $condition;
    }

    foreach ($condition->getContextDefinitions() as $name => $definition) {
      $matches = $this->contextHandler->getMatchingContexts(
        $this->contextRepository->getAvailableContexts(),
        $definition
      );

      if (empty($matches)) {
        continue;
      }

      $match = reset($matches);
      $condition->setContextValue($name, $match->getContextValue());
    }

    return $condition;
  }

  /**
   * Checks if the current user is authenticated via ACM.
   *
   * @return bool
   *   TRUE if the user is authenticated via ACM.
   */
  protected function isAcmAuthenticated(): bool {
    return $this->session->isAuthenticated() && $this->session->getRealmPluginId() === 'acm';
  }

  /**
   * Get the access menu settings.
   *
   * @param \Drupal\Core\Cache\CacheableMetadata $metadata
   *   The cacheable metadata.
   *
   * @return array
   *   The access menu settings.
   */
  protected function getAccessMenuSettings(CacheableMetadata $metadata): array {
    $profile = [
      'active' => $this->isAcmAuthenticated(),
      'loginUrl' => Url::fromRoute('oidc.openid_connect.login', ['realm' => 'acm'])->toString(),
      'loginRedirectUrl' => Url::fromRoute('oidc.openid_connect.login_redirect')->toString(),
      'logoutUrl' => Url::fromRoute('oidc.openid_connect.logout')->toString(),
    ];

    if ($this->isAcmAuthenticated()) {
      $metadata->addCacheContexts(['session']);

      // Switch capacity URL.
      $metadata->addCacheTags(['config:oidc.realm.acm']);
      $audiences = $this->configFactory->get('oidc.realm.acm')->get('audiences') ?? [];

      if (count($audiences) > 1) {
        $metadata->addCacheContexts(['url']);
        $profile['switchCapacityUrl'] = Url::fromRoute('oidc_mcpf.switch_capacity', [], [
          'query' => $this->getDestinationArray(),
        ])->toString();
      }

      // User data.
      $given_name = $this->session->getJsonWebTokens()->getGivenName();
      $family_name = $this->session->getJsonWebTokens()->getFamilyName();

      if (!empty($given_name) && !empty($family_name)) {
        $profile['idpData']['user'] = [
          'firstName' => $given_name,
          'name' => $family_name,
        ];
      }
    }

    return [
      'profile' => $profile,
      'visibility' => $this->currentUser->isAnonymous() || $this->isAcmAuthenticated(),
    ];
  }

  /**
   * Get the language switcher settings.
   *
   * @param \Drupal\Core\Cache\CacheableMetadata $metadata
   *   The cacheable metadata.
   *
   * @return array|null
   *   The language switcher settings.
   */
  protected function getLanguageSwitcherSettings(CacheableMetadata $metadata): ?array {
    if (!$this->configFactory->get('oidc_mcpf.settings')->get('toolbar.language_switcher')) {
      return NULL;
    }

    $metadata->addCacheContexts(['url']);

    // Get the language switch links.
    if ($this->pathMatcher->isFrontPage() || !$this->currentRouteMatch->getRouteObject()) {
      $url = Url::fromRoute('<front>');
    }
    else {
      $url = Url::fromRouteMatch($this->currentRouteMatch);
    }

    $language_switch_links = $this->languageManager->getLanguageSwitchLinks(
      LanguageInterface::TYPE_INTERFACE,
      $url
    );

    // No switcher if no links.
    if (empty($language_switch_links->links)) {
      return NULL;
    }

    // Only keep the links we need and leave if there are less than 2.
    $links = array_intersect_key($language_switch_links->links, ['en' => 0, 'nl' => 0, 'fr' => 0]);

    if (count($links) < 2) {
      return NULL;
    }

    // Return in [id => ['href' => url]] format.
    return array_map(fn (array $link) => ['href' => $link['url']->toString()], $links);
  }

}
