/**
 * @file
 * Javascript to switch between light and dark mode.
 *
 * More specifically, this script
 * - manages the flipping between moon/sun on the #theme-switch button.
 * - adds a light/dark class to the HTML tag.
 * - saves the new theme to a cookie available site-wide for 30 days.
 */
(function (Drupal, once) {
  /**
   * Attaches the theme mode toggle behaviour to the #theme-switch button.
   *
   * @type {Drupal~behavior}
   *
   * @prop {Drupal~behaviorAttach} attach
   *   Adds a click listener to the #theme-switch button that changes between
   *   light and dark mode when clicked. See the file docblock for the specifics
   *   of each action that entails.
   */
  Drupal.behaviors.toliveroDarkSwitchThemeModeToggle = {
    attach(context, drupalSettings) {

      // Centralize the name of the cookie used to track theme preference.
      const cookieName = 'olivero_dark_switch.theme';

      // 1. Grab the theme toggle button from the DOM.
      const button = document.querySelector("[data-theme-toggle]");

      // 2. Get the current site setting using the html data-theme attribute.
      let currentThemeSetting = document.querySelector('html').getAttribute('data-theme');
      console.log('Current theme setting is: ' + currentThemeSetting);

      // 3. Update the theme setting and button according to current settings.
      updateButton({ buttonEl: button, isDark: currentThemeSetting === "dark" });

      // 4. Apply the configuration settings.
      button.style.float = drupalSettings.oliveroDarkSwitch.float;

      // 5. Add an event listener to toggle the theme.
      button.addEventListener("click", (event) => {
        const newTheme = currentThemeSetting === "dark" ? "light" : "dark";
        console.log('Switching theme to: ' + newTheme);

        updateButton({ buttonEl: button, isDark: newTheme === "dark" });
        updateThemeOnHtmlEl({ theme: newTheme });
        setThemeCookie({ theme: newTheme });

        currentThemeSetting = newTheme;
      });

      /**
       * Update the aria-label and svg icon visibility base on theme mode.
       *
       * @param {Element} buttonEl
       *   The button to be updated (i.e. the #theme-switch button).
       * @param {bool} isDark
       *   TRUE if the current theme mode is "dark" and FALSE otherwise.
       *
       * @return {Element}
       *   The updated button element.
       */
      function updateButton({ buttonEl, isDark }) {
        const newAria = isDark ? "Change to light theme" : "Change to dark theme";

        // Use an aria-label to make it clear for screen readers.
        buttonEl.setAttribute("aria-label", newAria);

        // Indicate the inner icon based on the theme mode.
        updateSunMoonIcon({parentButton: buttonEl, isDark: isDark });

        return buttonEl;
      }

      /**
       * Updates visibility of both the sun and moon icons based on theme mode.
       *
       * @param {Element} parentButton
       *   The #theme-switch button element which contains the icons to update.
       * @param {bool} isDark
       *   TRUE if the current theme mode is "dark" and FALSE otherwise.
       *
       * @return {Element}
       *   The icon which is currently active. For example, if the theme mode is
       *   dark then the sun icon is returned, but if it is light then the
       *   moon icon is returned.
       */
      function updateSunMoonIcon({parentButton, isDark}) {
        let sunIcon = document.querySelector('#theme-switch-sun');
        let moonIcon = document.querySelector('#theme-switch-moon');

        // If we are using dark mode then the sun should be visible.
        if (isDark) {
          sunIcon.style.display = 'block';
          moonIcon.style.display = 'none';
          // Also add sun class to parent button for easy styling.
          parentButton.classList.add('sun');
          parentButton.classList.remove('moon');
        }
        // Otherwise, using light mode so the moon should be visible.
        else {
          sunIcon.style.display = 'none';
          moonIcon.style.display = 'block';
          // Also add moon class to parent button for easy styling.
          parentButton.classList.remove('sun');
          parentButton.classList.add('moon');
        }

        return isDark ? sunIcon : moonIcon;
      }

      /**
       * Updates the theme data attribute on the html tag.
       *
       * @param {string} theme
       *   Either "dark" if the theme mode is dark or "light" otherwise.
       *
       * @return {Element}
       *   The html tag element that was updated.
       */
      function updateThemeOnHtmlEl({ theme }) {
        let element  = document.querySelector("html");
        console.log('Updating html data-theme attribute to: ' + theme);
        element.setAttribute("data-theme", theme);
        return element;
      }

      /**
       * Sets a cookie to keep track of this user's theme preference.
       *
       * @param {string} theme
       *   Either "dark" if the theme mode is dark or "light" otherwise.
       *
       * @return {void}
       *   Simply sets the cookie; does not return anything.
       */
      function setThemeCookie({ theme }) {
        document.cookie = cookieName + '=' + theme +'; max-age=2592000; path=/; SameSite=Lax';
      }
    }
  };
}(Drupal, once));
