<?php

namespace Drupal\one_time_login_link_admin\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Config\ConfigManagerInterface;
use Drupal\Core\Routing\RedirectDestinationInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines OneTimeLoginLinkController Class.
 *
 * @package Drupal\one_time_login_link_admin\Controller
 */
class OneTimeLoginLinkController extends ControllerBase {

  /**
   * The Messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The redirect destination.
   *
   * @var \Drupal\Core\Routing\RedirectDestinationInterface
   */
  protected $redirectDestination;

  /**
   * Drupal\Core\Config\ConfigManagerInterface definition.
   *
   * @var \Drupal\Core\Config\ConfigManagerInterface
   */
  protected $configManager;

  /**
   * Constructs the service objects.
   */
  public function __construct(MessengerInterface $messenger,
                              RedirectDestinationInterface $redirect_destination,
                              ConfigManagerInterface $configfactory) {
    $this->messenger = $messenger;
    $this->redirectDestination = $redirect_destination;
    $this->configManager = $configfactory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('messenger'),
      $container->get('redirect.destination'),
      $container->get('config.manager')
    );
  }

  /**
   * Generates a one-time login (password reset) link for the given user.
   *
   * @param \Drupal\Core\Session\AccountInterface $user
   *   The user for which to generate the login link.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse|null
   *   A redirect to the destination, if one was provided.
   */
  public function generateLoginLink(AccountInterface $user) {
    $config = $this->configManager->getConfigFactory()->getEditable('user.settings');
    $timeout = $config->get('password_reset_timeout');
    $url = user_pass_reset_url($user);
    $mail = $user->getEmail();
    $this->messenger->addMessage($this->t('One-time login link created for %mail:<br/> <code>%login</code>', [
      '%mail' => $mail,
      '%login' => $url,
    ]));

    $this->messenger->addMessage($this->t("This link is valid for %hr, and will become invalid if the user's password is updated by anyone.", [
      '%hr' => $this->secTohr($timeout),
    ]));

    if ($destination = $this->redirectDestination->get()) {
      return new RedirectResponse($destination);
    }
  }

  /**
   * Generate second to human readable format.
   *
   * @int $seconds
   *   Time value in second.
   *
   * @return string
   *   A time string in human readable format.
   */
  public function secTohr($seconds): string {
    $response = "Invalid time";
    if (is_numeric($seconds)) {
      $hours = floor($seconds / 3600);
      $minutes = floor(($seconds / 60) % 60);
      $seconds = $seconds % 60;
      return $hours > 0 ? "$hours hours" : ($minutes > 0 ? "$minutes minutes, $seconds seconds" : "$seconds seconds");
    }
    return $response;
  }

}
