<?php

namespace Drupal\one_time_login_link_admin\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Routing\RedirectDestinationInterface;
use Drupal\one_time_login_link_admin\HelperService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines OneTimeLoginLinkController Class.
 *
 * @package Drupal\one_time_login_link_admin\Controller
 */
class OneTimeLoginLinkController extends ControllerBase {

  /**
   * The Messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The redirect destination.
   *
   * @var \Drupal\Core\Routing\RedirectDestinationInterface
   */
  protected $redirectDestination;

  /**
   * The helper service.
   *
   * @var \Drupal\one_time_login_link_admin\HelperService
   */
  protected $helperService;

  /**
   * Constructs the service objects.
   */
  public function __construct(MessengerInterface $messenger,
                              RedirectDestinationInterface $redirect_destination,
                              HelperService $helper_service) {
    $this->messenger = $messenger;
    $this->redirectDestination = $redirect_destination;
    $this->helperService = $helper_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('messenger'),
      $container->get('redirect.destination'),
      $container->get('one_time_login_link_admin.helper'),
    );
  }

  /**
   * Generates a one-time login (password reset) link for the given user.
   *
   * @param \Drupal\Core\Session\AccountInterface $user
   *   The user for which to generate the login link.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse|null
   *   A redirect to the destination, if one was provided.
   */
  public function generateLoginLink(AccountInterface $user) {
    $timeout = $this->config('user.settings')->get('password_reset_timeout');
    $url = user_pass_reset_url($user);
    $mail = $user->getEmail();
    $this->messenger->addMessage($this->t('One-time login link created for %mail:<br/> <code>%login</code>', [
      '%mail' => $mail,
      '%login' => $url,
    ]));

    $this->messenger->addMessage($this->t("This link is valid for %hr, and will become invalid if the user's password is updated by anyone.", [
      '%hr' => $this->helperService->secTohr($timeout),
    ]));

    if ($destination = $this->redirectDestination->get()) {
      return new RedirectResponse($destination);
    }
  }

}
