<?php

namespace Drupal\one_time_login_rest\Controller;

use Drupal\user\Controller\UserAuthenticationController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;

/**
 * Controller routines for One Time Login routes.
 */
class OneTimeLoginUserAuthenticationController extends UserAuthenticationController {

  /**
   * {@inheritdoc}
   */
  public function resetPassLogin(Request $request) {
    $current = \Drupal::time()->getRequestTime();
    $format = $this->getRequestFormat($request);

    $content = $request->getContent();
    $credentials = $this->serializer->decode($content, $format);

    $required_body = [
      'uid',
      'timestamp',
      'hash',
    ];

    $empty = [];
    foreach ($required_body as $field) {
      if (empty($credentials[$field])) {
        $empty[$field] = $field;
      }
    }

    if (!empty($empty)) {
      throw new BadRequestHttpException('Missing body: ' . implode(', ', $empty));
    }

    /** @var \Drupal\user\UserInterface $user */
    $user = $this->userStorage->load($credentials['uid']);

    $this->floodControl($request, $credentials['uid']);

    if (empty($user)) {
      throw new BadRequestHttpException('The user was not found.');
    }
    if (!empty($user) && $user->isBlocked()) {
      throw new BadRequestHttpException('The user has not been activated or is blocked.');
    }

    $timestamp = $credentials['timestamp'];

    // Time out, in seconds, until login URL expires.
    $timeout = $this->config('user.settings')->get('password_reset_timeout');
    // No time out for first time login.
    if ($user->getLastLoginTime() && $current - $timestamp > $timeout) {
      throw new BadRequestHttpException('You have tried to use a one-time login link that has expired. Please request a new one.');
    }
    elseif ($user->isAuthenticated() && ($timestamp >= $user->getLastLoginTime()) && ($timestamp <= $current) && hash_equals($credentials['hash'], user_pass_rehash($user, $timestamp))) {
      $this->userFloodControl->clear('user.http_login', $this->getLoginFloodIdentifier($request, $credentials['uid']));
      /** @var \Drupal\user\UserInterface $user */
      $this->userLoginFinalize($user);

      // Send basic metadata about the logged in user.
      $response_data = [];
      if ($user->get('uid')->access('view', $user)) {
        $response_data['current_user']['uid'] = $user->id();
      }
      if ($user->get('roles')->access('view', $user)) {
        $response_data['current_user']['roles'] = $user->getRoles();
      }
      if ($user->get('name')->access('view', $user)) {
        $response_data['current_user']['name'] = $user->getAccountName();
      }
      $response_data['csrf_token'] = $this->csrfToken->get('rest');

      $logout_route = $this->routeProvider->getRouteByName('user.logout.http');
      // Trim '/' off path to match \Drupal\Core\Access\CsrfAccessCheck.
      $logout_path = ltrim($logout_route->getPath(), '/');
      $response_data['logout_token'] = $this->csrfToken->get($logout_path);

      $encoded_response_data = $this->serializer->encode($response_data, $format);
      return new Response($encoded_response_data);
    }

    $flood_config = $this->config('user.flood');
    if ($identifier = $this->getLoginFloodIdentifier($request, $credentials['uid'])) {
      $this->userFloodControl->register('user.http_reset_login', $flood_config->get('user_window'), $identifier);
    }
    // Always register an IP-based failed login event.
    $this->userFloodControl->register('user.failed_login_ip', $flood_config->get('ip_window'));
    throw new BadRequestHttpException('Sorry, invalid one-time-login link.');
  }

}
