<?php

namespace Drupal\onelogin_integration;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Password\DefaultPasswordGenerator;
use Drupal\user\UserInterface;

/**
 * Class UserService for the OneLogin Integration module.
 *
 * Creates a user with the username / password coming from the OneLogin account
 * that the user uses to log in with. This service is triggered when someone is
 * trying to log in through OneLogin, but no account exists yet for the given
 * username / email.
 *
 * @package Drupal\onelogin_integration
 */
class UserService implements UserServiceInterface {

  /**
   * UserService constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Reference to TypeManagerInterface.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   Reference to MessengerInterface.
   * @param \Drupal\Core\Password\DefaultPasswordGenerator $passwordGenerator
   *   Reference to Password Generator.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected MessengerInterface $messenger,
    protected DefaultPasswordGenerator $passwordGenerator,
  ) {}

  /**
   * Creates a user.
   *
   * @param string $username
   *   The username for the new user.
   * @param string $email
   *   The email for the new user.
   *
   * @return \Drupal\user\UserInterface
   *   Returns a user object.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function createUser(string $username, string $email): UserInterface {
    $user = $this->entityTypeManager->getStorage('user')->create(
      [
        'name'                     => $username,
        'mail'                     => $email,
        'pass'                     => $this->passwordGenerator->generate(),
        'enforceIsNew'             => TRUE,
        'init'                     => $email,
        'defaultLangcode'          => 'en',
        'preferred_langcode'       => 'en',
        'preferred_admin_langcode' => 'en',
        'status'                   => 0,
      ]
    );

    $user->save();

    $this->messenger->addMessage("User with uid " . $user->id() . " saved!\n");

    return $user;
  }

}
