<?php

namespace Drupal\onelogin_integration;

use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\MissingDependencyException;
use Drupal\Core\Url;

/**
 * Class SamlAuthenticatorFactory.
 *
 * @package Drupal\onelogin_integration
 */
class SAMLAuthenticatorFactory implements SAMLAuthenticatorFactoryInterface {

  /**
   * The variable that holds an instance of ConfigFactoryInterface.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  private $configFactory;

  /**
   * SamlAuthenticatorFactory constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   Reference to ConfigFactoryInterface.
   *
   * @throws \Drupal\Core\Extension\MissingDependencyException
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->configFactory = $config_factory;

    // Check if OneLogin SAML library is installed.
    if (!class_exists('OneLogin_Saml2_Auth')) {
      throw new MissingDependencyException('The Onelogin Saml2 plugin is not correctly configured');
    }
  }

  /**
   * Settings for the OneLogin_Saml2_Auth library.
   *
   * Creates an instance of the OneLogin_Saml2_Auth library with default and,
   * if given, custom settings.
   *
   * @param array $settings
   *   Custom settings for the initialization of the OneLogin_Saml2_Auth
   *   library.
   *
   * @return \OneLogin_Saml2_Auth
   *   Returns a new instance of the OneLogin_Saml2_Auth library.
   */
  public function createFromSettings(array $settings = []) {
    $default_settings = [
      'strict' => $this->configFactory->get('onelogin_integration.settings')->get('strict_mode'),
      'debug' => $this->configFactory->get('onelogin_integration.settings')->get('debug'),

      'sp' => [
        'entityId' => $this->configFactory->get('onelogin_integration.settings')->get('sp_entity_id'),
        'assertionConsumerService' => [
          'url' => Url::fromRoute('onelogin_integration.acs', [], ['absolute' => TRUE])->toString(),
        ],
        'singleLogoutService' => [
          'url' => Url::fromRoute('onelogin_integration.slo', [], ['absolute' => TRUE])->toString(),
        ],
        'NameIDFormat' => $this->configFactory->get('onelogin_integration.settings')->get('nameid_format'),
        'x509cert' => $this->configFactory->get('onelogin_integration.settings')->get('sp_x509cert'),
        'privateKey' => $this->configFactory->get('onelogin_integration.settings')->get('sp_privatekey'),
      ],

      'idp' => [
        'entityId' => $this->configFactory->get('onelogin_integration.settings')->get('entityid'),
        'singleSignOnService' => [
          'url' => $this->configFactory->get('onelogin_integration.settings')->get('sso'),
        ],
        'singleLogoutService' => [
          'url' => $this->configFactory->get('onelogin_integration.settings')->get('slo'),
        ],
        'x509cert' => $this->configFactory->get('onelogin_integration.settings')->get('x509cert'),
      ],

      'security' => [
        'signMetadata' => FALSE,
        'nameIdEncrypted' => $this->configFactory->get('onelogin_integration.settings')->get('nameid_encrypted'),
        'authnRequestsSigned' => $this->configFactory->get('onelogin_integration.settings')->get('authn_request_signed'),
        'logoutRequestSigned' => $this->configFactory->get('onelogin_integration.settings')->get('logout_request_signed'),
        'logoutResponseSigned' => $this->configFactory->get('onelogin_integration.settings')->get('logout_response_signed'),
        'wantMessagesSigned' => $this->configFactory->get('onelogin_integration.settings')->get('want_message_signed'),
        'wantAssertionsSigned' => $this->configFactory->get('onelogin_integration.settings')->get('want_assertion_signed'),
        'wantAssertionsEncrypted' => $this->configFactory->get('onelogin_integration.settings')->get('want_assertion_encrypted'),
        'relaxDestinationValidation' => TRUE,
      ],
    ];

    $settings = NestedArray::mergeDeep($default_settings, $settings);

    return new \OneLogin_Saml2_Auth($settings);
  }

}
