<?php

namespace Drupal\onetimelogin\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Drupal\user\Entity\User;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Drupal\Component\Datetime\TimeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller class for generating a one-time login URL.
 */
class OneTimeLoginController extends ControllerBase {

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * Constructs a OneTimeLoginController object.
   *
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   */
  public function __construct(TimeInterface $time) {
    $this->time = $time;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('datetime.time')
    );
  }

  /**
   * Generates a one-time login URL for the given user.
   *
   * @param \Drupal\user\Entity\User $user
   *   The user entity for whom the login link is generated.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The HTTP request object.
   *
   * @return array
   *   A render array containing the one-time login link and information.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException
   *   Throws access denied exception if the user dosen't have permission.
   */
  public function generate(User $user, Request $request) {
    // Check if the current user has permission to generate one-time login link.
    if (!$this->currentUser()->hasPermission('access one-time login')) {
      throw new AccessDeniedHttpException();
    }

    // Get the current request time as a timestamp.
    $timestamp = $this->time->getRequestTime();

    // Generate a password reset link using the user entity and timestamp.
    $hash = user_pass_rehash($user, $timestamp);

    // Construct the internal URI for the one-time login URL.
    $uri = 'user/reset/' . $user->id() . '/' . $timestamp . '/' . $hash . '/login';

    // Generate an absolute URL from the internal URI.
    $url = Url::fromUri('internal:/' . $uri, ['absolute' => TRUE]);

    // Return a render array containing the login link and user information.
    return [
      '#markup' => '
        <div class="card">
          <div class="card-header">' . $this->t('One-Time Login URL') . '</div>
          <div class="card-body">
            <h6 class="card-title">' . $this->t('One-time login link for the user <strong>@username</strong>', ['@username' => $user->getAccountName()]) . '</h6>
            <p class="card-text">
              ' . $this->t('This one-time login link is valid for 24 hours from the time of generation. It should only be used by @username to securely access their account. Copy and paste this link in an Incognito Window.<br><br>For security reasons, please do not share this link with others.', ['@username' => $user->getAccountName()]) . '
            </p>
          </div>
          <div class="card-footer">
            <code>' . $url->toString() . '</code>
          </div>
        </div>',
      '#allowed_tags' => ['div', 'strong', 'h6', 'p', 'a', 'br', 'pre', 'code'],
    ];
  }

}
