<?php

namespace Drupal\onetimelogin\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Drupal\user\Entity\User;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Drupal\Component\Datetime\TimeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Psr\Log\LoggerInterface;

/**
 * Controller class for generating a one-time login URL.
 */
final class OneTimeLoginController extends ControllerBase {

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * Logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs a OneTimeLoginController object.
   *
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   */
  public function __construct(TimeInterface $time, LoggerInterface $logger) {
    $this->time = $time;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new self(
      $container->get('datetime.time'),
      $container->get('logger.factory')->get('onetimelogin')
    );
  }

  /**
   * Generates a one-time login URL for the given user.
   *
   * @param \Drupal\user\Entity\User $user
   *   The user entity for whom the login link is generated.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The HTTP request object.
   *
   * @return array
   *   A render array containing the one-time login link and information.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException
   *   Throws access denied exception if the user doesn't have permission.
   */
  public function generate(User $user, Request $request) {
    // Check if the current user has permission to generate one-time login link.
    if (!$this->currentUser()->hasPermission('access one-time login')) {
      $this->logger->warning('Unauthorized one-time login attempt by user @uid from IP @ip.', [
        '@uid' => $this->currentUser()->id(),
        '@ip' => $request->getClientIp(),
      ]);
      throw new AccessDeniedHttpException();
    }

    // Log the one-time login link generation.
    $this->logger->notice('One-time login link generated for user @uid by user @admin_uid.', [
      '@uid' => $user->id(),
      '@admin_uid' => $this->currentUser()->id(),
    ]);

    // Get the current request time as a timestamp.
    $timestamp = $this->time->getRequestTime();

    // Generate a password reset link using the user entity and timestamp.
    $hash = user_pass_rehash($user, $timestamp);

    // Generate an absolute URL for the one-time login route.
    $url = Url::fromRoute('user.reset.login', [
      'uid' => $user->id(),
      'timestamp' => $timestamp,
      'hash' => $hash,
    ], ['absolute' => TRUE]);

    return [
      '#type' => 'container',
      '#attributes' => ['class' => ['card']],
      'header' => [
        '#type' => 'html_tag',
        '#tag' => 'div',
        '#value' => $this->t('One-Time Login URL'),
        '#attributes' => ['class' => ['card-header']],
      ],
      'body' => [
        '#type' => 'container',
        '#attributes' => ['class' => ['card-body']],
        'title' => [
          '#type' => 'html_tag',
          '#tag' => 'h6',
          '#value' => $this->t('One-time login link for the user <strong>@username</strong>', ['@username' => $user->getAccountName()]),
          '#attributes' => ['class' => ['card-title']],
        ],
        'description' => [
          '#type' => 'html_tag',
          '#tag' => 'p',
          '#value' => $this->t('This one-time login link is valid for 24 hours from the time of generation. It should only be used by @username to securely access their account. Copy and paste this link in an Incognito Window.<br><br>For security reasons, please do not share this link with others.', ['@username' => $user->getAccountName()]),
          '#attributes' => ['class' => ['card-text']],
        ],
        'input_group' => [
          '#type' => 'container',
          '#attributes' => ['class' => ['mb-3']],
          'login_link' => [
            '#type' => 'textfield',
            '#title_display' => 'invisible',
            '#attributes' => [
              'id' => 'login-link',
              'class' => ['form-control'],
              'readonly' => 'readonly',
        // Add this to explicitly render value.
              'value' => $url->toString(),
            ],
            '#default_value' => $url->toString(),
          ],
          'copy_button' => [
            '#type' => 'button',
            '#value' => $this->t('Copy to Clipboard'),
            '#attributes' => [
              'id' => 'copy-button',
              'class' => ['btn', 'btn-primary', 'mt-2', 'btn-sm'],
            ],
          ],
          'generate_new_link' => [
            '#type' => 'link',
            '#title' => $this->t('Generate New URL'),
            '#url' => Url::fromRoute('<current>', ['user' => $user->id()]),
            '#attributes' => [
              'id' => 'generate-new-button',
              'class' => ['btn', 'btn-secondary', 'mt-2', 'btn-sm'],
            ],
          ],
        ],
      ],
      '#attached' => [
        'library' => [
          'onetimelogin/copy',
        ],
      ],
    ];
  }

}
