<?php

namespace Drupal\onetimelogin\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Drupal\user\Entity\User;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Drupal\Component\Datetime\TimeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\onetimelogin\Service\ShortUrlService;

/**
 * Controller class for generating a one-time login URL.
 */
final class OneTimeLoginController extends ControllerBase {

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * Logger service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $logger;

  /**
   * Short URL service.
   *
   * @var \Drupal\onetimelogin\Service\ShortUrlService
   */
  protected $shortUrlService;

  /**
   * Constructs a OneTimeLoginController object.
   *
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger factory service.
   * @param \Drupal\onetimelogin\Service\ShortUrlService $short_url_service
   *   The short URL service.
   */
  public function __construct(TimeInterface $time, LoggerChannelFactoryInterface $loggerFactory, ShortUrlService $short_url_service) {
    $this->time = $time;
    $this->logger = $loggerFactory->get('onetimelogin');
    $this->shortUrlService = $short_url_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new self(
      $container->get('datetime.time'),
      $container->get('logger.factory')->get('onetimelogin'),
      $container->get('onetimelogin.short_url_service')
    );
  }

  /**
   * Generates a one-time login URL for the given user.
   *
   * @param \Drupal\user\Entity\User $user
   *   The user entity for whom the login link is generated.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The HTTP request object.
   *
   * @return array
   *   A render array containing the one-time login link and information.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException
   *   Throws access denied exception if the user doesn't have permission.
   */
  public function generate(User $user, Request $request) {
    // Check if the current user has permission to generate one-time login link.
    if (!$this->currentUser()->hasPermission('access one-time login')) {
      $this->logger->warning('Unauthorized one-time login attempt by user @uid from IP @ip.', [
        '@uid' => $this->currentUser()->id(),
        '@ip' => $request->getClientIp(),
      ]);
      throw new AccessDeniedHttpException();
    }

    // Log the one-time login link generation.
    $this->logger->notice('One-time login link generated for user @uid by user @admin_uid.', [
      '@uid' => $user->id(),
      '@admin_uid' => $this->currentUser()->id(),
    ]);

    // Get the current request time as a timestamp.
    $timestamp = $this->time->getRequestTime();

    // Generate a password reset link using the user entity and timestamp.
    $hash = user_pass_rehash($user, $timestamp);

    // Generate an absolute URL for the one-time login route.
    $url = Url::fromRoute('user.reset.login', [
      'uid' => $user->id(),
      'timestamp' => $timestamp,
      'hash' => $hash,
    ], ['absolute' => TRUE]);

    $config = $this->config('onetimelogin.settings');
    $short_url = NULL;
    if ($config->get('enable_short_url')) {
      // Generate path for the short URL - use the actual reset login URL.
      $reset_url = Url::fromRoute('user.reset.login', [
        'uid' => $user->id(),
        'timestamp' => $timestamp,
        'hash' => $hash,
      ]);
      $path = $reset_url->getInternalPath();
      $hash_short = $this->shortUrlService->generateShortUrl($path);
      // Generate URL using the routing system.
      $options = ['absolute' => TRUE];
      $short_url = Url::fromRoute('onetimelogin.redirect', ['hash' => $hash_short], $options)->toString();
      $this->logger->info('Short URL generated for one-time login link: @short', ['@short' => $short_url]);

    }

    return [
      '#type' => 'container',
      '#attributes' => ['class' => ['card']],
      'header' => [
        '#type' => 'html_tag',
        '#tag' => 'div',
        '#value' => $this->t('One-Time Login URL'),
        '#attributes' => ['class' => ['card-header']],
      ],
      'body' => [
        '#type' => 'container',
        '#attributes' => ['class' => ['card-body']],
        'title' => [
          '#type' => 'html_tag',
          '#tag' => 'h6',
          '#value' => $this->t('One-time login link for the user <strong>@username</strong>', ['@username' => $user->getAccountName()]),
          '#attributes' => ['class' => ['card-title']],
        ],
        'description' => [
          '#type' => 'html_tag',
          '#tag' => 'p',
          '#value' => $this->t('This one-time login link is valid for 24 hours from the time of generation. It should only be used by @username to securely access their account. Copy and paste this link in an Incognito Window.<br><br>For security reasons, please do not share this link with others.', ['@username' => $user->getAccountName()]),
          '#attributes' => ['class' => ['card-text']],
        ],
        'full_url_label' => [
          '#type' => 'html_tag',
          '#tag' => 'label',
          '#value' => $this->t('Full One-Time Login URL:'),
          '#attributes' => ['class' => ['form-label', 'fw-bold', 'mt-3', 'mb-0']],
        ],
        'login_link' => [
          '#type' => 'textfield',
          '#title_display' => 'invisible',
          '#attributes' => [
            'id' => 'login-link',
            'class' => ['form-control'],
            'readonly' => 'readonly',
            'value' => $url->toString(),
          ],
          '#default_value' => $url->toString(),
        ],
        'short_url_label' => $short_url ? [
          '#type' => 'html_tag',
          '#tag' => 'label',
          '#value' => $this->t('Shortened URL:'),
          '#attributes' => ['class' => ['form-label', 'fw-bold', 'mt-4', 'mb-0']],
        ] : [],
        'short_link' => $short_url ? [
          '#type' => 'textfield',
          '#title_display' => 'invisible',
          '#attributes' => [
            'id' => 'short-link',
            'class' => ['form-control'],
            'readonly' => 'readonly',
            'value' => $short_url,
          ],
          '#default_value' => $short_url,
        ] : [],
        'action_buttons' => [
          '#type' => 'container',
          '#attributes' => ['class' => ['mt-4', 'd-flex', 'gap-2', 'flex-wrap']],
          'copy_full_button' => [
            '#type' => 'button',
            '#value' => $this->t('Copy Full URL'),
            '#attributes' => [
              'id' => 'copy-button',
              'class' => ['btn', 'btn-primary', 'btn-sm'],
            ],
          ],
          'copy_short_button' => $short_url ? [
            '#type' => 'button',
            '#value' => $this->t('Copy Shortened URL'),
            '#attributes' => [
              'id' => 'copy-short-button',
              'class' => ['btn', 'btn-success', 'btn-sm'],
              'type' => 'button',
            ],
          ] : [],
          'generate_new_link' => [
            '#type' => 'link',
            '#title' => $this->t('Generate New URL'),
            '#url' => Url::fromRoute('<current>', ['user' => $user->id()]),
            '#attributes' => [
              'id' => 'generate-new-button',
              'class' => ['btn', 'btn-secondary', 'btn-sm'],
            ],
          ],
        ],
      ],
      '#attached' => [
        'library' => [
          'onetimelogin/copy',
        ],
      ],
    ];
  }

}
