<?php

namespace Drupal\onetimelogin\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\onetimelogin\Service\ShortUrlService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Redirects short URLs to their full paths.
 */
class OneTimeLoginRedirectController extends ControllerBase {

  /**
   * The short URL service.
   *
   * @var \Drupal\onetimelogin\Service\ShortUrlService
   */
  protected ShortUrlService $shortUrlService;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Constructor.
   *
   * @param \Drupal\onetimelogin\Service\ShortUrlService $shortUrlService
   *   The short URL service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger factory service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(
    ShortUrlService $shortUrlService,
    LoggerChannelFactoryInterface $loggerFactory,
    MessengerInterface $messenger,
  ) {
    $this->shortUrlService = $shortUrlService;
    $this->loggerFactory = $loggerFactory;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('onetimelogin.short_url_service'),
      $container->get('logger.factory'),
      $container->get('messenger')
    );
  }

  /**
   * Handles redirect for short URLs.
   *
   * @param string $hash
   *   The short URL hash.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   A redirect response to the full path or home page.
   */
  public function handleRedirect($hash) {
    $path = $this->shortUrlService->getPath($hash);

    if ($path) {
      // Delete hash after use (one-time login).
      $this->shortUrlService->deleteHash($hash);

      $this->loggerFactory->get('onetimelogin')->notice(
        'One-time login redirect: @hash -> @path',
        ['@hash' => $hash, '@path' => $path]
      );

      return new RedirectResponse($path);
    }

    $this->messenger->addError($this->t('Invalid or expired short URL.'));

    return new RedirectResponse('/');
  }

}
