<?php

namespace Drupal\onetimelogin\Service;

use Drupal\Core\Database\Connection;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Provides short URL generation and management for one-time login.
 *
 * Generates cryptographically secure short URL hashes that map to full paths.
 * Each hash is stored in the database with an expiration time of 24 hours.
 */
class ShortUrlService {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $database;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected LoggerChannelFactoryInterface $loggerFactory;

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   Database service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   Logger factory service.
   */
  public function __construct(
    Connection $database,
    LoggerChannelFactoryInterface $loggerFactory,
  ) {
    $this->database = $database;
    $this->loggerFactory = $loggerFactory;
  }

  /**
   * Generate a secure short URL hash for a path.
   *
   * Uses cryptographically secure random bytes instead of MD5 hashing.
   * Each hash is stored in the database with a 24-hour expiration.
   *
   * @param string $path
   *   The full path to shorten (e.g., '/user/20/one-time-login').
   *
   * @return string
   *   The short URL hash (e.g., '26b6e75d').
   *
   * @throws \InvalidArgumentException
   *   If path is empty or invalid.
   * @throws \Exception
   *   If database insertion fails.
   */
  public function generateShortUrl(string $path): string {
    if (empty($path)) {
      throw new \InvalidArgumentException('Path cannot be empty.');
    }

    // Ensure path starts with forward slash for consistency.
    if (!str_starts_with($path, '/')) {
      $path = '/' . $path;
    }

    // Generate cryptographically secure hash (8 hex chars = 32 bits).
    $hash = bin2hex(random_bytes(4));

    try {
      $this->database->insert('onetimelogin_urls')
        ->fields([
          'hash' => $hash,
          'path' => $path,
          'created' => time(),
          'expires' => time() + (24 * 3600),
        ])
        ->execute();

      $this->loggerFactory->get('onetimelogin')->debug(
            'Generated short URL: @hash -> @path',
            ['@hash' => $hash, '@path' => $path]
        );
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('onetimelogin')->error(
            'Error generating short URL: @msg',
            ['@msg' => $e->getMessage()]
            );
      throw $e;
    }

    return $hash;
  }

  /**
   * Retrieve the path for a given hash.
   *
   * Validates the hash format and checks expiration. Expired hashes are
   * automatically deleted from the database.
   *
   * @param string $hash
   *   The short URL hash.
   *
   * @return string|null
   *   The full path if valid and not expired, NULL otherwise.
   */
  public function getPath(string $hash): ?string {
    if (!preg_match('/^[a-f0-9]{8}$/', $hash)) {
      $this->loggerFactory->get('onetimelogin')->warning(
            'Invalid hash format: @hash',
            ['@hash' => $hash]
        );
      return NULL;
    }

    $result = $this->database->select('onetimelogin_urls', 'o')
      ->fields('o', ['path', 'expires'])
      ->condition('hash', $hash)
      ->execute()
      ->fetch();

    if (!$result) {
      $this->loggerFactory->get('onetimelogin')->warning(
            'Short URL hash not found: @hash',
            ['@hash' => $hash]
        );
      return NULL;
    }

    // Check if expired.
    if ($result->expires < time()) {
      $this->database->delete('onetimelogin_urls')
        ->condition('hash', $hash)
        ->execute();

      $this->loggerFactory->get('onetimelogin')->warning(
            'Short URL hash expired: @hash',
            ['@hash' => $hash]
        );
      return NULL;
    }

    return $result->path;
  }

  /**
   * Delete a hash after use (one-time login).
   *
   * @param string $hash
   *   The short URL hash.
   */
  public function deleteHash(string $hash): void {
    $this->database->delete('onetimelogin_urls')
      ->condition('hash', $hash)
      ->execute();

    $this->loggerFactory->get('onetimelogin')->debug(
          'Deleted one-time login hash: @hash',
          ['@hash' => $hash]
      );
  }

}
