/**
 * @file
 * Copy to clipboard functionality for One-Time Login module.
 */
(function ($, Drupal, once) {
  Drupal.behaviors.copyToClipboard = {
    attach(context) {
      // Helper function to show a toast message (defined before use to satisfy ESLint)
      function showToast(message) {
        let container = document.querySelector('.toast-container');
        if (!container) {
          container = document.createElement('div');
          container.className =
            'toast-container position-fixed bottom-0 end-0 p-3';
          container.style.zIndex = '1080';
          document.body.appendChild(container);
        }

        const toastWrapper = document.createElement('div');
        toastWrapper.className = 'toast show';
        toastWrapper.setAttribute('role', 'alert');
        toastWrapper.setAttribute('aria-live', 'assertive');
        toastWrapper.setAttribute('aria-atomic', 'true');

        toastWrapper.innerHTML = `
          <div class="toast-header">
            <strong class="me-auto">OneTimeLogin</strong>
            <small class="text-muted">now</small>
            <button type="button" class="btn-close" data-bs-dismiss="toast" aria-label="Close"></button>
          </div>
          <div class="toast-body">${message}</div>
        `;

        container.appendChild(toastWrapper);

        // Auto-dismiss after 3 seconds
        setTimeout(() => {
          toastWrapper.classList.remove('show');
          toastWrapper.classList.add('hide');
          setTimeout(() => toastWrapper.remove(), 500);
        }, 3000);
      }

      // Fallback copy method (defined before use to satisfy ESLint)
      function fallbackCopyText(text) {
        const tempInput = document.createElement('input');
        tempInput.value = text;
        document.body.appendChild(tempInput);
        tempInput.select();
        tempInput.setSelectionRange(0, 99999); // For mobile devices
        try {
          document.execCommand('copy');
          showToast(Drupal.t('Link copied to clipboard!'));
        } catch (err) {
          console.error('Fallback copy failed: ', err);
          showToast(Drupal.t('Copy failed.'));
        }
        document.body.removeChild(tempInput);
      }

      once('copyToClipboard', '#copy-button', context).forEach((el) => {
        el.addEventListener('click', function (e) {
          e.preventDefault();
          const input = document.getElementById('login-link');
          if (input) {
            const text = input.value;

            if (navigator.clipboard && navigator.clipboard.writeText) {
              navigator.clipboard
                .writeText(text)
                .then(() =>
                  showToast(Drupal.t('Full URL copied to clipboard!')),
                )
                .catch((err) => {
                  console.error('Clipboard API failed: ', err);
                  fallbackCopyText(text);
                });
            } else {
              fallbackCopyText(text);
            }
          }
        });
      });

      // Handle copy for shortened URL button
      once('copyToClipboard', '#copy-short-button', context).forEach((el) => {
        el.addEventListener('click', function (e) {
          e.preventDefault();

          // Try multiple selectors to find the short URL input
          const input =
            document.getElementById('short-link') ||
            document.querySelector('input[name*="short"]') ||
            document.querySelector('#short-url-field') ||
            document.querySelector('.short-url-input');

          console.log('Short copy button clicked');
          console.log('Input element found:', input);

          if (!input) {
            // Debug: List all text inputs to help identify the correct one
            const allInputs = document.querySelectorAll('input[type="text"]');
            console.log('All text inputs found:', allInputs);
            allInputs.forEach((inp, index) => {
              console.log(`Input ${index}:`, inp.id, inp.name, inp.value);
            });
          }

          if (input) {
            const text = input.value;
            console.log('Text to copy:', text);

            if (navigator.clipboard && navigator.clipboard.writeText) {
              navigator.clipboard
                .writeText(text)
                .then(() => {
                  console.log('Clipboard API: Copy successful');
                  showToast(Drupal.t('Shortened URL copied to clipboard!'));
                })
                .catch((err) => {
                  console.error('Clipboard API failed: ', err);
                  fallbackCopyText(text);
                });
            } else {
              fallbackCopyText(text);
            }
          } else {
            console.log('Short URL input element not found');
            showToast(Drupal.t('Could not find shortened URL field'));
          }
        });
      });
    },
  };
})(jQuery, Drupal, once);
