<?php

namespace Drupal\onetimelogin\Commands;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Url;
use Drupal\onetimelogin\Service\ShortUrlService;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Mail\MailManagerInterface;
use Drupal\Core\Database\Connection;
use Drush\Commands\DrushCommands;

/**
 * Drush commands for generating one-time login links.
 */
class OneTimeLoginCommands extends DrushCommands {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * The short URL service.
   *
   * @var \Drupal\onetimelogin\Service\ShortUrlService
   */
  protected $shortUrlService;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The mail manager.
   *
   * @var \Drupal\Core\Mail\MailManagerInterface
   */
  protected $mailManager;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * Constructs a OneTimeLoginCommands object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Drupal\onetimelogin\Service\ShortUrlService $short_url_service
   *   The short URL service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Mail\MailManagerInterface $mail_manager
   *   The mail manager.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    TimeInterface $time,
    ShortUrlService $short_url_service,
    ConfigFactoryInterface $config_factory,
    LoggerChannelFactoryInterface $logger_factory,
    MailManagerInterface $mail_manager,
    Connection $database,
  ) {
    parent::__construct();
    $this->entityTypeManager = $entity_type_manager;
    $this->time = $time;
    $this->shortUrlService = $short_url_service;
    $this->configFactory = $config_factory;
    $this->loggerFactory = $logger_factory;
    $this->mailManager = $mail_manager;
    $this->database = $database;
  }

  /**
   * Generate a one-time login link for a user.
   *
   * @param string $user
   *   User ID or username.
   * @param array $options
   *   Command options.
   *
   * @command onetimelogin:generate
   * @aliases otl,otl-generate,one-time-login
   * @usage drush otl 123
   *   Generate one-time login link for user ID 123.
   * @usage drush otl john.doe
   *   Generate one-time login link for username 'john.doe'.
   * @usage drush otl admin --no-short
   *   Generate full URL only without short URL.
   * @validate-module-enabled onetimelogin
   */
  public function generate($user, array $options = ['no-short' => FALSE]) {
    // Load user by ID or username.
    $user_entity = NULL;

    if (is_numeric($user)) {
      $user_entity = $this->entityTypeManager->getStorage('user')->load($user);
    }
    else {
      $users = $this->entityTypeManager->getStorage('user')->loadByProperties(['name' => $user]);
      $user_entity = reset($users);
    }

    if (!$user_entity) {
      $this->logger()->error(dt('User "@user" not found.', ['@user' => $user]));
      return DrushCommands::EXIT_FAILURE;
    }

    // Validate user is active.
    if (!$user_entity->isActive()) {
      $this->logger()->error(
            dt(
                'User "@name" (UID: @uid) is blocked and cannot receive a login link.', [
                  '@name' => $user_entity->getAccountName(),
                  '@uid' => $user_entity->id(),
                ]
            )
        );
      return DrushCommands::EXIT_FAILURE;
    }

    // Validate user has email.
    if (!$user_entity->getEmail()) {
      $this->logger()->error(
            dt(
                'User "@name" (UID: @uid) does not have an email address.', [
                  '@name' => $user_entity->getAccountName(),
                  '@uid' => $user_entity->id(),
                ]
            )
        );
      return DrushCommands::EXIT_FAILURE;
    }

    // Generate the one-time login link.
    $timestamp = $this->time->getRequestTime();
    $hash = user_pass_rehash($user_entity, $timestamp);

    $url = Url::fromRoute(
          'user.reset.login', [
            'uid' => $user_entity->id(),
            'timestamp' => $timestamp,
            'hash' => $hash,
          ], ['absolute' => TRUE]
      );

    $full_url = $url->toString();

    // Display user information.
    $this->output()->writeln('');
    $this->output()->writeln('<info>One-Time Login Link Generated</info>');
    $this->output()->writeln('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
    $this->output()->writeln(
          dt(
              'User: @name (UID: @uid)', [
                '@name' => $user_entity->getAccountName(),
                '@uid' => $user_entity->id(),
              ]
          )
      );
    $this->output()->writeln(dt('Email: @email', ['@email' => $user_entity->getEmail()]));

    // Get expiration time from config.
    $config = $this->configFactory->get('onetimelogin.settings');
    $expiration_seconds = $config->get('link_expiration') ?? 86400;
    $expiration_hours = $expiration_seconds / 3600;
    $this->output()->writeln(dt('Expires in: @hours hours', ['@hours' => $expiration_hours]));
    $this->output()->writeln('');

    // Display full URL.
    $this->output()->writeln('<comment>Full URL:</comment>');
    $this->output()->writeln($full_url);

    // Generate and display short URL if enabled.
    if (!$options['no-short'] && $config->get('enable_short_url')) {
      try {
        $reset_url = Url::fromRoute(
              'user.reset.login', [
                'uid' => $user_entity->id(),
                'timestamp' => $timestamp,
                'hash' => $hash,
              ]
          );
        $path = $reset_url->getInternalPath();
        $hash_short = $this->shortUrlService->generateShortUrl($path);
        $short_url = Url::fromRoute('onetimelogin.redirect', ['hash' => $hash_short], ['absolute' => TRUE])->toString();

        $this->output()->writeln('');
        $this->output()->writeln('<comment>Short URL:</comment>');
        $this->output()->writeln($short_url);

        $this->loggerFactory->get('onetimelogin')->info(
              'One-time login link generated via Drush for user @uid. Short URL: @short', [
                '@uid' => $user_entity->id(),
                '@short' => $short_url,
              ]
          );
      }
      catch (\Exception $e) {
        $this->logger()->warning(dt('Failed to generate short URL: @msg', ['@msg' => $e->getMessage()]));
      }
    }
    else {
      $this->loggerFactory->get('onetimelogin')->info(
            'One-time login link generated via Drush for user @uid.', [
              '@uid' => $user_entity->id(),
            ]
        );
    }

    // Send email notification if available.
    $mail_sent = $this->mailManager->mail(
          'onetimelogin',
          'link_generated',
          $user_entity->getEmail(),
          $user_entity->getPreferredLangcode(),
          [
            'user' => $user_entity,
            'admin_name' => 'Drush CLI',
            'admin_uid' => 0,
          ]
      );

    if ($mail_sent['result']) {
      $this->output()->writeln('');
      $this->output()->writeln('<info>✓ Email notification sent to user</info>');
    }

    $this->output()->writeln('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
    $this->output()->writeln('');
    $this->output()->writeln('<comment>⚠ Security Notes:</comment>');
    $this->output()->writeln('  • This link can only be used ONCE');
    $this->output()->writeln('  • Share via secure channel only');
    $this->output()->writeln('  • User will be notified via email');
    $this->output()->writeln('');

    return DrushCommands::EXIT_SUCCESS;
  }

  /**
   * List active one-time login links.
   *
   * @param array $options
   *   Command options.
   *
   * @command onetimelogin:list
   * @aliases otl-list
   * @option limit Number of links to display (default: 10)
   * @option show-expired Include expired links in the list
   * @usage drush otl-list
   *   List last 10 active one-time login links.
   * @usage drush otl-list --limit=20
   *   List last 20 active links.
   * @usage drush otl-list --show-expired
   *   List all links including expired ones.
   * @validate-module-enabled onetimelogin
   */
  public function listLinks(array $options = ['limit' => 10, 'show-expired' => FALSE]) {
    $database = $this->database;

    $query = $database->select('onetimelogin_urls', 'o')
      ->fields('o', ['id', 'hash', 'created', 'expires', 'used', 'used_at', 'used_by_ip'])
      ->orderBy('created', 'DESC')
      ->range(0, $options['limit']);

    if (!$options['show-expired']) {
      $query->condition('expires', time(), '>=');
    }

    $results = $query->execute()->fetchAll();

    if (empty($results)) {
      $this->output()->writeln('<comment>No one-time login links found.</comment>');
      return DrushCommands::EXIT_SUCCESS;
    }

    $rows = [];
    $rows[] = ['ID', 'Hash', 'Created', 'Expires', 'Status', 'Used At', 'Used By IP'];

    foreach ($results as $row) {
      $status = $row->used ? 'Used' : ($row->expires < time() ? 'Expired' : 'Active');
      $rows[] = [
        $row->id,
        $row->hash,
        date('Y-m-d H:i', $row->created),
        date('Y-m-d H:i', $row->expires),
        $status,
        $row->used_at ? date('Y-m-d H:i', $row->used_at) : '-',
        $row->used_by_ip ?? '-',
      ];
    }

    $this->output()->writeln('');
    $this->io()->table($rows[0], array_slice($rows, 1));
    $this->output()->writeln('');

    return DrushCommands::EXIT_SUCCESS;
  }

  /**
   * Clean up expired one-time login links.
   *
   * @command onetimelogin:cleanup
   * @aliases otl-cleanup
   * @usage drush otl-cleanup
   *   Delete all expired one-time login links.
   * @validate-module-enabled onetimelogin
   */
  public function cleanup() {
    $database = $this->database;

    $deleted_count = $database->delete('onetimelogin_urls')
      ->condition('expires', time(), '<')
      ->execute();

    if ($deleted_count > 0) {
      $this->logger()->success(dt('Deleted @count expired one-time login link(s).', ['@count' => $deleted_count]));
      $this->loggerFactory->get('onetimelogin')->info(
            'Manual cleanup via Drush: Deleted @count expired link(s).', [
              '@count' => $deleted_count,
            ]
        );
    }
    else {
      $this->output()->writeln('<info>No expired links to clean up.</info>');
    }

    return DrushCommands::EXIT_SUCCESS;
  }

  /**
   * Check status of a one-time login link.
   *
   * @param string $hash
   *   The hash to check.
   *
   * @command onetimelogin:check
   * @aliases otl-check,otl-status
   * @usage drush otl-check a1b2c3d4e5f6
   *   Check the status of link with hash a1b2c3d4e5f6.
   * @validate-module-enabled onetimelogin
   */
  public function check($hash) {
    $database = $this->database;
    $current_time = $this->time->getRequestTime();

    // Query link details.
    $link = $database->select('onetimelogin_urls', 'o')
      ->fields('o')
      ->condition('hash', $hash)
      ->execute()
      ->fetchObject();

    if (!$link) {
      $this->logger()->error(dt('Link with hash "@hash" not found.', ['@hash' => $hash]));
      return DrushCommands::EXIT_FAILURE;
    }

    // Determine status.
    $status = '';
    $status_color = '';

    if ($link->used) {
      $status = 'USED';
      $status_color = 'error';
    }
    elseif ($link->expires < $current_time) {
      $status = 'EXPIRED';
      $status_color = 'comment';
    }
    else {
      $status = 'ACTIVE';
      $status_color = 'info';
    }

    // Display details.
    $this->output()->writeln('');
    $this->output()->writeln('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
    $this->output()->writeln('<comment>🔍 One-Time Login Link Status</comment>');
    $this->output()->writeln('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
    $this->output()->writeln('');

    $this->output()->writeln('  <comment>Hash:</comment>        ' . $link->hash);
    $this->output()->writeln('  <comment>Status:</comment>      <' . $status_color . '>' . $status . '</' . $status_color . '>');
    $this->output()->writeln('  <comment>Created:</comment>     ' . date('Y-m-d H:i:s', $link->created));
    $this->output()->writeln('  <comment>Expires:</comment>     ' . date('Y-m-d H:i:s', $link->expires));

    if ($link->used) {
      $this->output()->writeln('  <comment>Used At:</comment>     ' . date('Y-m-d H:i:s', $link->used_at));
      $this->output()->writeln('  <comment>Used By IP:</comment>  ' . ($link->used_by_ip ?: 'Unknown'));
    }
    else {
      $time_remaining = $link->expires - $current_time;
      if ($time_remaining > 0) {
        $hours = floor($time_remaining / 3600);
        $minutes = floor(($time_remaining % 3600) / 60);
        $this->output()->writeln('  <comment>Time Left:</comment>   ' . $hours . 'h ' . $minutes . 'm');
      }
    }

    $this->output()->writeln('');
    $this->output()->writeln('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
    $this->output()->writeln('');

    return DrushCommands::EXIT_SUCCESS;
  }

}
