<?php

namespace Drupal\onetimelogin\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\onetimelogin\Service\ShortUrlService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Redirects short URLs to their full paths.
 */
class OneTimeLoginRedirectController extends ControllerBase {

  /**
   * The short URL service.
   *
   * @var \Drupal\onetimelogin\Service\ShortUrlService
   */
  protected ShortUrlService $shortUrlService;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Constructor.
   *
   * @param \Drupal\onetimelogin\Service\ShortUrlService $shortUrlService
   *   The short URL service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger factory service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(
    ShortUrlService $shortUrlService,
    LoggerChannelFactoryInterface $loggerFactory,
    MessengerInterface $messenger,
  ) {
    $this->shortUrlService = $shortUrlService;
    $this->loggerFactory = $loggerFactory;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('onetimelogin.short_url_service'),
      $container->get('logger.factory'),
      $container->get('messenger')
    );
  }

  /**
   * Handles redirect for short URLs.
   *
   * @param string $hash
   *   The short URL hash.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The HTTP request object.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   A redirect response to the full path or home page.
   */
  public function handleRedirect($hash, Request $request) {
    $ip_address = $request->getClientIp();
    $user_agent = $request->headers->get('User-Agent', 'Unknown');
    $path = $this->shortUrlService->getPath($hash, $ip_address);

    if ($path) {
      $this->loggerFactory->get('onetimelogin')->notice(
        'One-time login redirect successful: @hash -> @path from IP @ip, User-Agent: @ua',
        [
          '@hash' => $hash,
          '@path' => $path,
          '@ip' => $ip_address,
          '@ua' => $user_agent,
        ]
      );

      return new RedirectResponse($path);
    }

    $this->loggerFactory->get('onetimelogin')->warning(
      'One-time login redirect failed: Invalid or expired hash @hash from IP @ip, User-Agent: @ua',
      [
        '@hash' => $hash,
        '@ip' => $ip_address,
        '@ua' => $user_agent,
      ]
    );

    $this->messenger->addError($this->t('Invalid, expired, or already used one-time login link.'));

    return new RedirectResponse('/');
  }

}
