<?php

declare(strict_types=1);

namespace Drupal\onetimelogin\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Component\Datetime\TimeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for one-time login statistics.
 */
class OneTimeLoginStatisticsController extends ControllerBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * Constructs a OneTimeLoginStatisticsController object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   */
  public function __construct(Connection $database, TimeInterface $time) {
    $this->database = $database;
    $this->time = $time;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database'),
      $container->get('datetime.time')
    );
  }

  /**
   * Display one-time login statistics.
   *
   * @return array
   *   A render array.
   */
  public function view(): array {
    $current_time = $this->time->getRequestTime();

    // Overall statistics.
    $total_generated = $this->database->select('onetimelogin_urls', 'o')
      ->countQuery()
      ->execute()
      ->fetchField();

    $total_used = $this->database->select('onetimelogin_urls', 'o')
      ->condition('used', 1)
      ->countQuery()
      ->execute()
      ->fetchField();

    $total_expired = $this->database->select('onetimelogin_urls', 'o')
      ->condition('used', 0)
      ->condition('expires', $current_time, '<')
      ->countQuery()
      ->execute()
      ->fetchField();

    $total_active = $this->database->select('onetimelogin_urls', 'o')
      ->condition('used', 0)
      ->condition('expires', $current_time, '>=')
      ->countQuery()
      ->execute()
      ->fetchField();

    // Today's statistics.
    $today_start = strtotime('today');

    $today_generated = $this->database->select('onetimelogin_urls', 'o')
      ->condition('created', $today_start, '>=')
      ->countQuery()
      ->execute()
      ->fetchField();

    $today_used = $this->database->select('onetimelogin_urls', 'o')
      ->condition('used', 1)
      ->condition('used_at', $today_start, '>=')
      ->countQuery()
      ->execute()
      ->fetchField();

    // Last 7 days trend.
    $week_ago = $current_time - (7 * 24 * 60 * 60);

    $week_generated = $this->database->select('onetimelogin_urls', 'o')
      ->condition('created', $week_ago, '>=')
      ->countQuery()
      ->execute()
      ->fetchField();

    $week_used = $this->database->select('onetimelogin_urls', 'o')
      ->condition('used', 1)
      ->condition('used_at', $week_ago, '>=')
      ->countQuery()
      ->execute()
      ->fetchField();

    // Recent used links.
    $recent_used = $this->database->select('onetimelogin_urls', 'o')
      ->fields('o', ['hash', 'used_at', 'used_by_ip'])
      ->condition('used', 1)
      ->orderBy('used_at', 'DESC')
      ->range(0, 10)
      ->execute()
      ->fetchAll();

    $build = [];

    // Overview section.
    $build['overview'] = [
      '#type' => 'details',
      '#title' => $this->t('Overview'),
      '#open' => TRUE,
    ];

    $build['overview']['stats'] = [
      '#theme' => 'item_list',
      '#items' => [
        $this->t('Total links generated: <strong>@count</strong>', ['@count' => $total_generated]),
        $this->t('Total links used: <strong>@count</strong>', ['@count' => $total_used]),
        $this->t('Currently active: <strong>@count</strong>', ['@count' => $total_active]),
        $this->t('Expired (unused): <strong>@count</strong>', ['@count' => $total_expired]),
      ],
    ];

    // Usage rate calculation.
    if ($total_generated > 0) {
      $usage_rate = round(($total_used / $total_generated) * 100, 1);
      $build['overview']['usage_rate'] = [
        '#markup' => '<p>' . $this->t('Usage rate: <strong>@rate%</strong>', ['@rate' => $usage_rate]) . '</p>',
      ];
    }

    // Today's activity.
    $build['today'] = [
      '#type' => 'details',
      '#title' => $this->t("Today's Activity"),
      '#open' => TRUE,
    ];

    $build['today']['stats'] = [
      '#theme' => 'item_list',
      '#items' => [
        $this->t('Generated today: <strong>@count</strong>', ['@count' => $today_generated]),
        $this->t('Used today: <strong>@count</strong>', ['@count' => $today_used]),
      ],
    ];

    // Last 7 days.
    $build['week'] = [
      '#type' => 'details',
      '#title' => $this->t('Last 7 Days'),
      '#open' => TRUE,
    ];

    $build['week']['stats'] = [
      '#theme' => 'item_list',
      '#items' => [
        $this->t('Generated: <strong>@count</strong>', ['@count' => $week_generated]),
        $this->t('Used: <strong>@count</strong>', ['@count' => $week_used]),
      ],
    ];

    // Recent usage.
    if (!empty($recent_used)) {
      $build['recent'] = [
        '#type' => 'details',
        '#title' => $this->t('Recent Usage (Last 10)'),
        '#open' => TRUE,
      ];

      $rows = [];
      foreach ($recent_used as $record) {
        $rows[] = [
          $record->hash,
          date('Y-m-d H:i:s', $record->used_at),
          $record->used_by_ip ?: $this->t('Unknown'),
        ];
      }

      $build['recent']['table'] = [
        '#type' => 'table',
        '#header' => [
          $this->t('Hash'),
          $this->t('Used At'),
          $this->t('IP Address'),
        ],
        '#rows' => $rows,
      ];
    }

    return $build;
  }

}
