<?php

declare(strict_types=1);

namespace Drupal\onetimelogin\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Routing\UrlGeneratorInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Controller for OpenAPI documentation.
 */
class OpenApiController extends ControllerBase {

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The URL generator.
   *
   * @var \Drupal\Core\Routing\UrlGeneratorInterface
   */
  protected $urlGenerator;

  /**
   * Constructs an OpenApiController object.
   *
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   * @param \Drupal\Core\Routing\UrlGeneratorInterface $url_generator
   *   The URL generator.
   */
  public function __construct(RequestStack $request_stack, UrlGeneratorInterface $url_generator) {
    $this->requestStack = $request_stack;
    $this->urlGenerator = $url_generator;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('request_stack'),
      $container->get('url_generator')
    );
  }

  /**
   * Generates OpenAPI specification.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The OpenAPI specification in JSON format.
   */
  public function spec(): JsonResponse {
    $request = $this->requestStack->getCurrentRequest();
    $base_url = $request->getSchemeAndHttpHost();

    $spec = [
      'openapi' => '3.0.0',
      'info' => [
        'title' => 'One-Time Login API',
        'description' => 'REST API for managing one-time login links in Drupal',
        'version' => '1.0.0',
        'contact' => [
          'name' => 'API Support',
        ],
        'license' => [
          'name' => 'GPL-2.0-or-later',
        ],
      ],
      'servers' => [
        [
          'url' => $base_url . '/api/v1',
          'description' => 'Production server',
        ],
      ],
      'paths' => [
        '/onetimelogin/generate' => [
          'post' => [
            'summary' => 'Generate one-time login link',
            'description' => 'Creates a new one-time login link for a user',
            'operationId' => 'generateLink',
            'tags' => ['One-Time Login'],
            'security' => [
              ['basicAuth' => []],
              ['cookieAuth' => []],
            ],
            'requestBody' => [
              'required' => TRUE,
              'content' => [
                'application/json' => [
                  'schema' => [
                    'type' => 'object',
                    'properties' => [
                      'uid' => [
                        'type' => 'integer',
                        'description' => 'User ID',
                        'example' => 123,
                      ],
                      'username' => [
                        'type' => 'string',
                        'description' => 'Username (alternative to uid)',
                        'example' => 'john.doe',
                      ],
                      'send_email' => [
                        'type' => 'boolean',
                        'description' => 'Send email notification',
                        'default' => TRUE,
                      ],
                    ],
                  ],
                  'examples' => [
                    'by_uid' => [
                      'summary' => 'Generate by user ID',
                      'value' => [
                        'uid' => 123,
                        'send_email' => TRUE,
                      ],
                    ],
                    'by_username' => [
                      'summary' => 'Generate by username',
                      'value' => [
                        'username' => 'john.doe',
                        'send_email' => FALSE,
                      ],
                    ],
                  ],
                ],
              ],
            ],
            'responses' => [
              '201' => [
                'description' => 'Link generated successfully',
                'content' => [
                  'application/json' => [
                    'schema' => [
                      '$ref' => '#/components/schemas/GenerateResponse',
                    ],
                  ],
                ],
              ],
              '400' => ['$ref' => '#/components/responses/BadRequest'],
              '403' => ['$ref' => '#/components/responses/Forbidden'],
              '404' => ['$ref' => '#/components/responses/NotFound'],
            ],
          ],
        ],
        '/onetimelogin/check/{hash}' => [
          'get' => [
            'summary' => 'Check link status',
            'description' => 'Retrieves the status of a one-time login link',
            'operationId' => 'checkLink',
            'tags' => ['One-Time Login'],
            'security' => [
              ['basicAuth' => []],
              ['cookieAuth' => []],
            ],
            'parameters' => [
              [
                'name' => 'hash',
                'in' => 'path',
                'required' => TRUE,
                'description' => '12-character hash of the link',
                'schema' => [
                  'type' => 'string',
                  'pattern' => '^[a-f0-9]{12}$',
                  'example' => 'a1b2c3d4e5f6',
                ],
              ],
            ],
            'responses' => [
              '200' => [
                'description' => 'Link status retrieved',
                'content' => [
                  'application/json' => [
                    'schema' => [
                      '$ref' => '#/components/schemas/CheckResponse',
                    ],
                  ],
                ],
              ],
              '403' => ['$ref' => '#/components/responses/Forbidden'],
              '404' => ['$ref' => '#/components/responses/NotFound'],
            ],
          ],
        ],
        '/onetimelogin/revoke/{hash}' => [
          'delete' => [
            'summary' => 'Revoke link',
            'description' => 'Revokes an active one-time login link',
            'operationId' => 'revokeLink',
            'tags' => ['One-Time Login'],
            'security' => [
              ['basicAuth' => []],
              ['cookieAuth' => []],
            ],
            'parameters' => [
              [
                'name' => 'hash',
                'in' => 'path',
                'required' => TRUE,
                'description' => '12-character hash of the link',
                'schema' => [
                  'type' => 'string',
                  'pattern' => '^[a-f0-9]{12}$',
                  'example' => 'a1b2c3d4e5f6',
                ],
              ],
            ],
            'responses' => [
              '200' => [
                'description' => 'Link revoked successfully',
                'content' => [
                  'application/json' => [
                    'schema' => [
                      '$ref' => '#/components/schemas/RevokeResponse',
                    ],
                  ],
                ],
              ],
              '403' => ['$ref' => '#/components/responses/Forbidden'],
              '404' => ['$ref' => '#/components/responses/NotFound'],
            ],
          ],
        ],
        '/onetimelogin/list' => [
          'get' => [
            'summary' => 'List links',
            'description' => 'Retrieves a paginated list of one-time login links',
            'operationId' => 'listLinks',
            'tags' => ['One-Time Login'],
            'security' => [
              ['basicAuth' => []],
              ['cookieAuth' => []],
            ],
            'parameters' => [
              [
                'name' => 'status',
                'in' => 'query',
                'description' => 'Filter by status',
                'schema' => [
                  'type' => 'string',
                  'enum' => ['active', 'expired', 'used', 'all'],
                  'default' => 'all',
                ],
              ],
              [
                'name' => 'limit',
                'in' => 'query',
                'description' => 'Number of results per page',
                'schema' => [
                  'type' => 'integer',
                  'minimum' => 1,
                  'maximum' => 100,
                  'default' => 50,
                ],
              ],
              [
                'name' => 'offset',
                'in' => 'query',
                'description' => 'Pagination offset',
                'schema' => [
                  'type' => 'integer',
                  'minimum' => 0,
                  'default' => 0,
                ],
              ],
            ],
            'responses' => [
              '200' => [
                'description' => 'Links retrieved successfully',
                'content' => [
                  'application/json' => [
                    'schema' => [
                      '$ref' => '#/components/schemas/ListResponse',
                    ],
                  ],
                ],
              ],
              '403' => ['$ref' => '#/components/responses/Forbidden'],
            ],
          ],
        ],
        '/onetimelogin/statistics' => [
          'get' => [
            'summary' => 'Get statistics',
            'description' => 'Retrieves usage statistics for one-time login links',
            'operationId' => 'getStatistics',
            'tags' => ['One-Time Login'],
            'security' => [
              ['basicAuth' => []],
              ['cookieAuth' => []],
            ],
            'responses' => [
              '200' => [
                'description' => 'Statistics retrieved successfully',
                'content' => [
                  'application/json' => [
                    'schema' => [
                      '$ref' => '#/components/schemas/StatisticsResponse',
                    ],
                  ],
                ],
              ],
              '403' => ['$ref' => '#/components/responses/Forbidden'],
            ],
          ],
        ],
      ],
      'components' => [
        'securitySchemes' => [
          'basicAuth' => [
            'type' => 'http',
            'scheme' => 'basic',
            'description' => 'Basic HTTP authentication with Drupal username and password',
          ],
          'cookieAuth' => [
            'type' => 'apiKey',
            'in' => 'cookie',
            'name' => 'SESS*',
            'description' => 'Session cookie authentication (for same-site requests)',
          ],
        ],
        'schemas' => $this->getSchemas(),
        'responses' => [
          'BadRequest' => [
            'description' => 'Bad request',
            'content' => [
              'application/json' => [
                'schema' => ['$ref' => '#/components/schemas/Error'],
              ],
            ],
          ],
          'Forbidden' => [
            'description' => 'Forbidden - insufficient permissions',
            'content' => [
              'application/json' => [
                'schema' => ['$ref' => '#/components/schemas/Error'],
              ],
            ],
          ],
          'NotFound' => [
            'description' => 'Resource not found',
            'content' => [
              'application/json' => [
                'schema' => ['$ref' => '#/components/schemas/Error'],
              ],
            ],
          ],
        ],
      ],
      'tags' => [
        [
          'name' => 'One-Time Login',
          'description' => 'Operations for managing one-time login links',
        ],
      ],
    ];

    return new JsonResponse($spec);
  }

  /**
   * Displays Swagger UI.
   *
   * @return array
   *   Render array for Swagger UI.
   */
  public function ui(): array {
    return [
      '#theme' => 'onetimelogin_swagger_ui',
      '#attached' => [
        'library' => [
          'onetimelogin/swagger-ui',
        ],
        'drupalSettings' => [
          'onetimelogin' => [
            'apiSpecUrl' => $this->urlGenerator->generateFromRoute('onetimelogin.api.spec'),
          ],
        ],
      ],
    ];
  }

  /**
   * Gets component schemas for OpenAPI spec.
   *
   * @return array
   *   Component schemas.
   */
  protected function getSchemas(): array {
    return [
      'GenerateResponse' => [
        'type' => 'object',
        'properties' => [
          'success' => ['type' => 'boolean', 'example' => TRUE],
          'data' => [
            'type' => 'object',
            'properties' => [
              'hash' => ['type' => 'string', 'example' => 'a1b2c3d4e5f6'],
              'short_url' => ['type' => 'string', 'example' => 'https://example.com/s/a1b2c3d4e5f6'],
              'full_url' => ['type' => 'string', 'example' => 'https://example.com/user/reset/123/...'],
              'user' => [
                'type' => 'object',
                'properties' => [
                  'uid' => ['type' => 'integer', 'example' => 123],
                  'username' => ['type' => 'string', 'example' => 'john.doe'],
                  'email' => ['type' => 'string', 'example' => 'john@example.com'],
                ],
              ],
              'expires' => ['type' => 'integer', 'example' => 1738109856],
              'expires_human' => ['type' => 'string', 'example' => '2026-01-29 12:02:56'],
              'email_sent' => ['type' => 'boolean', 'example' => TRUE],
            ],
          ],
          'message' => ['type' => 'string', 'example' => 'One-time login link generated successfully.'],
        ],
      ],
      'CheckResponse' => [
        'type' => 'object',
        'properties' => [
          'success' => ['type' => 'boolean', 'example' => TRUE],
          'data' => [
            'type' => 'object',
            'properties' => [
              'hash' => ['type' => 'string', 'example' => 'a1b2c3d4e5f6'],
              'status' => ['type' => 'string', 'enum' => ['active', 'expired', 'used'], 'example' => 'active'],
              'created' => ['type' => 'integer', 'example' => 1738023456],
              'created_human' => ['type' => 'string', 'example' => '2026-01-28 12:02:56'],
              'expires' => ['type' => 'integer', 'example' => 1738109856],
              'expires_human' => ['type' => 'string', 'example' => '2026-01-29 12:02:56'],
              'time_remaining' => ['type' => 'integer', 'example' => 86400],
              'time_remaining_human' => ['type' => 'string', 'example' => '23h 48m'],
              'used_at' => ['type' => 'integer', 'example' => 1738025000],
              'used_at_human' => ['type' => 'string', 'example' => '2026-01-28 12:30:00'],
              'used_by_ip' => ['type' => 'string', 'example' => '192.168.1.100'],
            ],
          ],
        ],
      ],
      'RevokeResponse' => [
        'type' => 'object',
        'properties' => [
          'success' => ['type' => 'boolean', 'example' => TRUE],
          'data' => [
            'type' => 'object',
            'properties' => [
              'hash' => ['type' => 'string', 'example' => 'a1b2c3d4e5f6'],
              'revoked_by' => [
                'type' => 'object',
                'properties' => [
                  'uid' => ['type' => 'integer', 'example' => 1],
                  'username' => ['type' => 'string', 'example' => 'admin'],
                ],
              ],
              'revoked_at' => ['type' => 'integer', 'example' => 1738023600],
              'revoked_at_human' => ['type' => 'string', 'example' => '2026-01-28 12:06:40'],
            ],
          ],
          'message' => ['type' => 'string', 'example' => 'Link revoked successfully.'],
        ],
      ],
      'ListResponse' => [
        'type' => 'object',
        'properties' => [
          'success' => ['type' => 'boolean', 'example' => TRUE],
          'data' => [
            'type' => 'object',
            'properties' => [
              'links' => [
                'type' => 'array',
                'items' => [
                  'type' => 'object',
                  'properties' => [
                    'id' => ['type' => 'integer', 'example' => 767],
                    'hash' => ['type' => 'string', 'example' => 'a1b2c3d4e5f6'],
                    'status' => ['type' => 'string', 'enum' => ['active', 'expired', 'used']],
                    'created' => ['type' => 'integer'],
                    'created_human' => ['type' => 'string'],
                    'expires' => ['type' => 'integer'],
                    'expires_human' => ['type' => 'string'],
                  ],
                ],
              ],
              'pagination' => [
                'type' => 'object',
                'properties' => [
                  'total' => ['type' => 'integer', 'example' => 150],
                  'limit' => ['type' => 'integer', 'example' => 50],
                  'offset' => ['type' => 'integer', 'example' => 0],
                  'count' => ['type' => 'integer', 'example' => 50],
                ],
              ],
            ],
          ],
        ],
      ],
      'StatisticsResponse' => [
        'type' => 'object',
        'properties' => [
          'success' => ['type' => 'boolean', 'example' => TRUE],
          'data' => [
            'type' => 'object',
            'properties' => [
              'overall' => [
                'type' => 'object',
                'properties' => [
                  'total_generated' => ['type' => 'integer', 'example' => 1250],
                  'total_used' => ['type' => 'integer', 'example' => 980],
                  'total_active' => ['type' => 'integer', 'example' => 45],
                  'total_expired' => ['type' => 'integer', 'example' => 225],
                  'usage_rate' => ['type' => 'number', 'format' => 'float', 'example' => 78.4],
                ],
              ],
              'today' => [
                'type' => 'object',
                'properties' => [
                  'generated' => ['type' => 'integer', 'example' => 35],
                  'used' => ['type' => 'integer', 'example' => 28],
                ],
              ],
              'last_7_days' => [
                'type' => 'object',
                'properties' => [
                  'generated' => ['type' => 'integer', 'example' => 187],
                  'used' => ['type' => 'integer', 'example' => 145],
                ],
              ],
            ],
          ],
        ],
      ],
      'Error' => [
        'type' => 'object',
        'properties' => [
          'error' => [
            'type' => 'object',
            'properties' => [
              'code' => ['type' => 'integer', 'example' => 403],
              'message' => ['type' => 'string', 'example' => 'Insufficient permissions'],
            ],
          ],
        ],
      ],
    ];
  }

}
