<?php

declare(strict_types=1);

namespace Drupal\onetimelogin\Form;

use Drupal\Core\Database\Connection;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\user\UserInterface;
use Drupal\onetimelogin\Service\ShortUrlService;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Provides a confirmation form for revoking one-time login links.
 */
class RevokeForm extends ConfirmFormBase
{

    /**
     * The user entity.
     *
     * @var \Drupal\user\UserInterface
     */
    protected $user;

    /**
     * The short URL service.
     *
     * @var \Drupal\onetimelogin\Service\ShortUrlService
     */
    protected $shortUrlService;

    /**
     * The request stack.
     *
     * @var \Symfony\Component\HttpFoundation\RequestStack
     */
    protected $requestStack;

    /**
     * The database connection.
     *
     * @var \Drupal\Core\Database\Connection
     */
    protected $database;

    /**
     * The time service.
     *
     * @var \Drupal\Component\Datetime\TimeInterface
     */
    protected $time;

    /**
     * The logger service.
     *
     * @var \Psr\Log\LoggerInterface
     */
    protected $logger;

    /**
     * Constructs a RevokeForm object.
     *
     * @param \Drupal\onetimelogin\Service\ShortUrlService   $short_url_service
     *   The short URL service.
     * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
     *   The request stack.
     * @param \Drupal\Core\Database\Connection               $database
     *   The database connection.
     * @param \Drupal\Component\Datetime\TimeInterface       $time
     *   The time service.
     * @param \Psr\Log\LoggerInterface                       $logger
     *   The logger service.
     */
    public function __construct(
        ShortUrlService $short_url_service,
        RequestStack $request_stack,
        Connection $database,
        TimeInterface $time,
        LoggerInterface $logger
    ) {
        $this->shortUrlService = $short_url_service;
        $this->requestStack = $request_stack;
        $this->database = $database;
        $this->time = $time;
        $this->logger = $logger;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container)
    {
        return new static(
            $container->get('onetimelogin.short_url_service'),
            $container->get('request_stack'),
            $container->get('database'),
            $container->get('datetime.time'),
            $container->get('logger.channel.onetimelogin')
        );
    }

    /**
     * {@inheritdoc}
     */
    public function getFormId(): string
    {
        return 'onetimelogin_revoke_form';
    }

    /**
     * {@inheritdoc}
     */
    public function buildForm(array $form, FormStateInterface $form_state, ?UserInterface $user = null): array
    {
        $this->user = $user;

        // Get active links for this user.
        $current_time = $this->time->getRequestTime();

        $active_links = $this->database->select('onetimelogin_urls', 'o')
            ->fields('o', ['hash', 'created', 'expires'])
            ->condition('path', '%/user/' . $user->id() . '/%', 'LIKE')
            ->condition('used', 0)
            ->condition('expires', $current_time, '>=')
            ->execute()
            ->fetchAll();

        if (empty($active_links)) {
            $this->messenger()->addWarning(
                $this->t(
                    'No active one-time login links found for user @name.', [
                    '@name' => $user->getDisplayName(),
                    ]
                )
            );
            return [
            '#markup' => $this->t('No links to revoke.'),
            ];
        }

        // Build table of active links.
        $rows = [];
        foreach ($active_links as $link) {
            $rows[] = [
            $link->hash,
            date('Y-m-d H:i:s', $link->created),
            date('Y-m-d H:i:s', $link->expires),
            ];
        }

        $form['links_table'] = [
        '#type' => 'table',
        '#header' => [
        $this->t('Hash'),
        $this->t('Created'),
        $this->t('Expires'),
        ],
        '#rows' => $rows,
        '#prefix' => '<p>' . $this->t('The following active one-time login links will be revoked:') . '</p>',
        ];

        $form['count'] = [
        '#type' => 'value',
        '#value' => count($active_links),
        ];

        $form['hashes'] = [
        '#type' => 'value',
        '#value' => array_column($active_links, 'hash'),
        ];

        return parent::buildForm($form, $form_state);
    }

    /**
     * {@inheritdoc}
     */
    public function getQuestion()
    {
        return $this->t(
            'Are you sure you want to revoke all active one-time login links for @name?', [
            '@name' => $this->user->getDisplayName(),
            ]
        );
    }

    /**
     * {@inheritdoc}
     */
    public function getCancelUrl()
    {
        return new Url('entity.user.collection');
    }

    /**
     * {@inheritdoc}
     */
    public function getDescription()
    {
        return $this->t('This action will immediately invalidate all active links. This cannot be undone.');
    }

    /**
     * {@inheritdoc}
     */
    public function getConfirmText()
    {
        return $this->t('Revoke All Links');
    }

    /**
     * {@inheritdoc}
     */
    public function submitForm(array &$form, FormStateInterface $form_state)
    {
        $current_user = $this->currentUser();
        $request = $this->requestStack->getCurrentRequest();
        $ip_address = $request ? $request->getClientIp() : null;

        $hashes = $form_state->getValue('hashes');
        $revoked_count = 0;

        foreach ($hashes as $hash) {
            if ($this->shortUrlService->revokeLink($hash, (int) $current_user->id(), $ip_address)) {
                $revoked_count++;
            }
        }

        if ($revoked_count > 0) {
            $this->messenger()->addStatus(
                $this->formatPlural(
                    $revoked_count,
                    'Successfully revoked 1 one-time login link for @name.',
                    'Successfully revoked @count one-time login links for @name.',
                    [
                    '@name' => $this->user->getDisplayName(),
                    '@count' => $revoked_count,
                    ]
                )
            );

            $this->logger->notice(
                'Admin UID @admin_uid revoked @count one-time login link(s) for user UID @user_uid from IP @ip',
                [
                '@admin_uid' => $current_user->id(),
                '@count' => $revoked_count,
                '@user_uid' => $this->user->id(),
                '@ip' => $ip_address ?? 'unknown',
                ]
            );
        }

        $form_state->setRedirectUrl($this->getCancelUrl());
    }

}
