<?php

declare(strict_types=1);

namespace Drupal\onetimelogin\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure onetimelogin settings for this site.
 */
final class Settings extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'onetimelogin_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['onetimelogin.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('onetimelogin.settings');

    $form['short_url_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Short URL Settings'),
      '#open' => TRUE,
    ];

    $form['short_url_settings']['enable_short_url'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Short URL Generation'),
      '#default_value' => $config->get('enable_short_url'),
      '#description' => $this->t('When enabled, a short URL will be generated automatically for one-time login links.'),
    ];

    $form['expiration_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Link Expiration Settings'),
      '#open' => TRUE,
    ];

    $form['expiration_settings']['link_expiration'] = [
      '#type' => 'select',
      '#title' => $this->t('Link Expiration Time'),
      '#default_value' => $config->get('link_expiration') ?? 86400,
      '#options' => [
        3600 => $this->t('1 hour'),
        21600 => $this->t('6 hours'),
        43200 => $this->t('12 hours'),
        86400 => $this->t('24 hours (default)'),
      ],
      '#description' => $this->t('How long one-time login links remain valid before expiration.'),
    ];

    $form['cleanup_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Automatic Cleanup Settings'),
      '#open' => TRUE,
    ];

    $form['cleanup_settings']['enable_cron_cleanup'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable automatic cleanup via cron'),
      '#default_value' => $config->get('enable_cron_cleanup') ?? TRUE,
      '#description' => $this->t('When enabled, expired links will be automatically deleted during cron runs. Disable if you want to keep expired links for audit purposes.'),
    ];

    $form['cleanup_settings']['cleanup_retention'] = [
      '#type' => 'select',
      '#title' => $this->t('Retention period for expired links'),
      '#default_value' => $config->get('cleanup_retention') ?? 0,
      '#options' => [
        0 => $this->t('Delete immediately after expiration'),
        86400 => $this->t('Keep for 1 day after expiration'),
        259200 => $this->t('Keep for 3 days after expiration'),
        604800 => $this->t('Keep for 7 days after expiration'),
        2592000 => $this->t('Keep for 30 days after expiration'),
      ],
      '#description' => $this->t('How long to keep expired links before deletion. Useful for audit trails and security investigations.'),
      '#states' => [
        'visible' => [
          ':input[name="enable_cron_cleanup"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['rate_limit_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Rate Limiting Settings'),
      '#open' => TRUE,
    ];

    $form['rate_limit_settings']['enable_rate_limit'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable rate limiting for link generation'),
      '#default_value' => $config->get('enable_rate_limit') ?? TRUE,
      '#description' => $this->t('Limit how many links can be generated per user within a time window. Helps prevent abuse and DoS attacks.'),
    ];

    $form['rate_limit_settings']['rate_limit_count'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum links per time window'),
      '#default_value' => $config->get('rate_limit_count') ?? 10,
      '#min' => 1,
      '#max' => 100,
      '#description' => $this->t('Maximum number of links a user can generate within the time window.'),
      '#states' => [
        'visible' => [
          ':input[name="enable_rate_limit"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['rate_limit_settings']['rate_limit_window'] = [
      '#type' => 'select',
      '#title' => $this->t('Time window'),
      '#default_value' => $config->get('rate_limit_window') ?? 60,
      '#options' => [
        30 => $this->t('30 seconds'),
        60 => $this->t('1 minute'),
        300 => $this->t('5 minutes'),
        600 => $this->t('10 minutes'),
        3600 => $this->t('1 hour'),
      ],
      '#description' => $this->t('Time window for rate limit counting.'),
      '#states' => [
        'visible' => [
          ':input[name="enable_rate_limit"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('onetimelogin.settings')
      ->set('enable_short_url', $form_state->getValue('enable_short_url'))
      ->set('link_expiration', $form_state->getValue('link_expiration'))
      ->set('enable_cron_cleanup', $form_state->getValue('enable_cron_cleanup'))
      ->set('cleanup_retention', $form_state->getValue('cleanup_retention'))
      ->set('enable_rate_limit', $form_state->getValue('enable_rate_limit'))
      ->set('rate_limit_count', $form_state->getValue('rate_limit_count'))
      ->set('rate_limit_window', $form_state->getValue('rate_limit_window'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
