<?php

declare(strict_types=1);

namespace Drupal\onetimelogin\Plugin\rest\resource;

use Drupal\Core\Database\Connection;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\rest\Plugin\ResourceBase;
use Drupal\rest\ResourceResponse;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Provides a resource to check one-time login link status.
 *
 * @RestResource(
 *   id = "onetimelogin_check",
 *   label = @Translation("Check One-Time Login Link"),
 *   uri_paths = {
 *     "canonical" = "/api/v1/onetimelogin/check/{hash}"
 *   }
 * )
 */
class CheckResource extends ResourceBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * Constructs a CheckResource object.
   *
   * @param array $configuration
   *   Configuration array.
   * @param string $plugin_id
   *   Plugin ID.
   * @param mixed $plugin_definition
   *   Plugin definition.
   * @param array $serializer_formats
   *   Serializer formats.
   * @param \Psr\Log\LoggerInterface $logger
   *   Logger service.
   * @param \Drupal\Core\Database\Connection $database
   *   Database connection.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   Current user.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   Time service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    array $serializer_formats,
    LoggerInterface $logger,
    Connection $database,
    AccountProxyInterface $current_user,
    TimeInterface $time,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $serializer_formats, $logger);
    $this->database = $database;
    $this->currentUser = $current_user;
    $this->time = $time;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->getParameter('serializer.formats'),
      $container->get('logger.factory')->get('onetimelogin'),
      $container->get('database'),
      $container->get('current_user'),
      $container->get('datetime.time')
    );
  }

  /**
   * Responds to GET requests.
   *
   * @param string $hash
   *   The hash to check.
   *
   * @return \Drupal\rest\ResourceResponse
   *   The response.
   */
  public function get(string $hash): ResourceResponse {
    // Check permission.
    if (!$this->currentUser->hasPermission('access one-time login') &&
        !$this->currentUser->hasPermission('view one-time login statistics')) {
      throw new AccessDeniedHttpException('Insufficient permissions to check link status.');
    }

    // Query link details.
    $link = $this->database->select('onetimelogin_urls', 'o')
      ->fields('o')
      ->condition('hash', $hash)
      ->execute()
      ->fetchObject();

    if (!$link) {
      throw new NotFoundHttpException('Link not found.');
    }

    $current_time = $this->time->getRequestTime();

    // Determine status.
    $status = 'active';
    if ($link->used) {
      $status = 'used';
    }
    elseif ($link->expires < $current_time) {
      $status = 'expired';
    }

    // Calculate time remaining.
    $time_remaining = NULL;
    if ($status === 'active') {
      $time_remaining = $link->expires - $current_time;
    }

    // Build response.
    $response_data = [
      'success' => TRUE,
      'data' => [
        'hash' => $link->hash,
        'status' => $status,
        'created' => (int) $link->created,
        'created_human' => date('Y-m-d H:i:s', $link->created),
        'expires' => (int) $link->expires,
        'expires_human' => date('Y-m-d H:i:s', $link->expires),
      ],
    ];

    if ($link->used) {
      $response_data['data']['used_at'] = (int) $link->used_at;
      $response_data['data']['used_at_human'] = date('Y-m-d H:i:s', $link->used_at);
      $response_data['data']['used_by_ip'] = $link->used_by_ip ?: NULL;
    }

    if ($time_remaining !== NULL) {
      $response_data['data']['time_remaining'] = $time_remaining;
      $response_data['data']['time_remaining_human'] = sprintf(
        '%dh %dm',
        floor($time_remaining / 3600),
        floor(($time_remaining % 3600) / 60)
      );
    }

    return new ResourceResponse($response_data, 200);
  }

}
