<?php

declare(strict_types=1);

namespace Drupal\onetimelogin\Plugin\rest\resource;

use Drupal\Core\Mail\MailManagerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\State\StateInterface;
use Drupal\user\UserAuthInterface;
use Drupal\onetimelogin\Service\ShortUrlService;
use Drupal\rest\Plugin\ResourceBase;
use Drupal\rest\ResourceResponse;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Provides a resource to generate one-time login links.
 *
 * @RestResource(
 *   id = "onetimelogin_generate",
 *   label = @Translation("Generate One-Time Login Link"),
 *   uri_paths = {
 *     "create" = "/api/v1/onetimelogin/generate"
 *   }
 * )
 */
class GenerateResource extends ResourceBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The short URL service.
   *
   * @var \Drupal\onetimelogin\Service\ShortUrlService
   */
  protected $shortUrlService;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The mail manager service.
   *
   * @var \Drupal\Core\Mail\MailManagerInterface
   */
  protected $mailManager;

  /**
   * The user auth service.
   *
   * @var \Drupal\user\UserAuthInterface
   */
  protected $userAuth;

  /**
   * Constructs a GenerateResource object.
   *
   * @param array $configuration
   *   Configuration array.
   * @param string $plugin_id
   *   Plugin ID.
   * @param mixed $plugin_definition
   *   Plugin definition.
   * @param array $serializer_formats
   *   Serializer formats.
   * @param \Psr\Log\LoggerInterface $logger
   *   Logger service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   Entity type manager.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   Current user.
   * @param \Drupal\onetimelogin\Service\ShortUrlService $short_url_service
   *   Short URL service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   Config factory service.
   * @param \Drupal\Core\State\StateInterface $state
   *   State service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   Request stack service.
   * @param \Drupal\Core\Mail\MailManagerInterface $mail_manager
   *   Mail manager service.
   * @param \Drupal\user\UserAuthInterface $user_auth
   *   User auth service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    array $serializer_formats,
    LoggerInterface $logger,
    EntityTypeManagerInterface $entity_type_manager,
    AccountProxyInterface $current_user,
    ShortUrlService $short_url_service,
    ConfigFactoryInterface $config_factory,
    StateInterface $state,
    RequestStack $request_stack,
    MailManagerInterface $mail_manager,
    UserAuthInterface $user_auth
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $serializer_formats, $logger);
    $this->entityTypeManager = $entity_type_manager;
    $this->currentUser = $current_user;
    $this->shortUrlService = $short_url_service;
    $this->configFactory = $config_factory;
    $this->state = $state;
    $this->requestStack = $request_stack;
    $this->mailManager = $mail_manager;
    $this->userAuth = $user_auth;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->getParameter('serializer.formats'),
      $container->get('logger.factory')->get('onetimelogin'),
      $container->get('entity_type.manager'),
      $container->get('current_user'),
      $container->get('onetimelogin.short_url_service'),
      $container->get('config.factory'),
      $container->get('state'),
      $container->get('request_stack'),
      $container->get('plugin.manager.mail'),
      $container->get('user.auth')
    );
  }

  /**
   * Responds to POST requests.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Drupal\rest\ResourceResponse
   *   The response.
   */
  public function post(Request $request): ResourceResponse {
    // Check permission.
    if (!$this->currentUser->hasPermission('access one-time login')) {
      throw new AccessDeniedHttpException('Insufficient permissions to generate one-time login links.');
    }

    // Parse JSON body.
    $data = json_decode($request->getContent(), TRUE);
    if (json_last_error() !== JSON_ERROR_NONE) {
      throw new BadRequestHttpException('Invalid JSON payload.');
    }

    // Validate required field.
    if (empty($data['uid']) && empty($data['username'])) {
      throw new BadRequestHttpException('Either "uid" or "username" is required.');
    }

    // Load user.
    $user = NULL;
    if (!empty($data['uid'])) {
      $user = $this->entityTypeManager->getStorage('user')->load($data['uid']);
    }
    elseif (!empty($data['username'])) {
      $users = $this->entityTypeManager->getStorage('user')->loadByProperties(['name' => $data['username']]);
      $user = reset($users);
    }

    if (!$user) {
      throw new NotFoundHttpException('User not found.');
    }

    // Validate user is active.
    if (!$user->isActive()) {
      throw new BadRequestHttpException('User is blocked and cannot receive login links.');
    }

    // Validate user has email.
    if (empty($user->getEmail())) {
      throw new BadRequestHttpException('User must have an email address.');
    }

    // Check rate limiting.
    $config = $this->configFactory->get('onetimelogin.settings');
    if ($config->get('enable_rate_limit') !== FALSE) {
      $rate_limit_window = $config->get('rate_limit_window') ?? 60;
      $max_requests = $config->get('rate_limit_count') ?? 10;
      $request = $this->requestStack->getCurrentRequest();
      $current_time = $request ? $request->getRequestTime() : time();

      $key = 'onetimelogin_api_rate_limit_' . $this->currentUser->id();
      $requests = $this->state->get($key, []);

      // Remove old entries.
      $requests = array_filter($requests, function ($timestamp) use ($current_time, $rate_limit_window) {
        return ($current_time - $timestamp) < $rate_limit_window;
      });

      if (count($requests) >= $max_requests) {
        throw new AccessDeniedHttpException('Rate limit exceeded. Please try again later.');
      }

      $requests[] = $current_time;
      $this->state->set($key, $requests);
    }

    // Generate one-time login URL.
    $link_expiration = $config->get('link_expiration') ?? 86400;
    $expiry = $current_time + $link_expiration;

    $reset_url = $this->userAuth->getResetUrl($user->id(), $current_time);

    // Generate short URL.
    $short_hash = $this->shortUrlService->generateShortUrl($reset_url, $expiry);
    $request = $this->requestStack->getCurrentRequest();
    $host = $request ? $request->getSchemeAndHttpHost() : 'http://localhost';
    $short_url = $host . '/s/' . $short_hash;

    // Send email notification if requested.
    $send_email = $data['send_email'] ?? TRUE;
    $email_sent = FALSE;

    if ($send_email) {
      $mail_result = $this->mailManager->mail(
        'onetimelogin',
        'link_generated',
        $user->getEmail(),
        $user->getPreferredLangcode(),
        [
          'user' => $user,
          'admin_name' => $this->currentUser->getDisplayName(),
          'admin_uid' => $this->currentUser->id(),
        ]
      );
      $email_sent = $mail_result['result'] ?? FALSE;
    }

    // Log the generation.
    $this->logger->info(
      'One-time login link generated via API for user @uid by API user @api_uid',
      [
        '@uid' => $user->id(),
        '@api_uid' => $this->currentUser->id(),
      ]
    );

    // Return response.
    $response_data = [
      'success' => TRUE,
      'data' => [
        'hash' => $short_hash,
        'short_url' => $short_url,
        'full_url' => $reset_url,
        'user' => [
          'uid' => (int) $user->id(),
          'username' => $user->getAccountName(),
          'email' => $user->getEmail(),
        ],
        'expires' => $expiry,
        'expires_human' => date('Y-m-d H:i:s', $expiry),
        'email_sent' => $email_sent,
      ],
      'message' => 'One-time login link generated successfully.',
    ];

    return new ResourceResponse($response_data, 201);
  }

}
