<?php

declare(strict_types=1);

namespace Drupal\onetimelogin\Plugin\rest\resource;

use Drupal\Core\Database\Connection;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\rest\Plugin\ResourceBase;
use Drupal\rest\ResourceResponse;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;

/**
 * Provides a resource to list one-time login links.
 *
 * @RestResource(
 *   id = "onetimelogin_list",
 *   label = @Translation("List One-Time Login Links"),
 *   uri_paths = {
 *     "canonical" = "/api/v1/onetimelogin/list"
 *   }
 * )
 */
class ListResource extends ResourceBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * Constructs a ListResource object.
   *
   * @param array $configuration
   *   Configuration array.
   * @param string $plugin_id
   *   Plugin ID.
   * @param mixed $plugin_definition
   *   Plugin definition.
   * @param array $serializer_formats
   *   Serializer formats.
   * @param \Psr\Log\LoggerInterface $logger
   *   Logger service.
   * @param \Drupal\Core\Database\Connection $database
   *   Database connection.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   Current user.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   Time service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    array $serializer_formats,
    LoggerInterface $logger,
    Connection $database,
    AccountProxyInterface $current_user,
    TimeInterface $time,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $serializer_formats, $logger);
    $this->database = $database;
    $this->currentUser = $current_user;
    $this->time = $time;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->getParameter('serializer.formats'),
      $container->get('logger.factory')->get('onetimelogin'),
      $container->get('database'),
      $container->get('current_user'),
      $container->get('datetime.time')
    );
  }

  /**
   * Responds to GET requests.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Drupal\rest\ResourceResponse
   *   The response.
   */
  public function get(Request $request): ResourceResponse {
    // Check permission.
    if (!$this->currentUser->hasPermission('view one-time login statistics')) {
      throw new AccessDeniedHttpException('Insufficient permissions to list links.');
    }

    $current_time = $this->time->getRequestTime();

    // Get query parameters.
    // active, expired, used, all.
    $status = $request->query->get('status');
    $limit = (int) $request->query->get('limit', 50);
    $offset = (int) $request->query->get('offset', 0);

    // Validate limit.
    if ($limit < 1 || $limit > 100) {
      $limit = 50;
    }

    // Build query.
    $query = $this->database->select('onetimelogin_urls', 'o')
      ->fields('o')
      ->orderBy('created', 'DESC')
      ->range($offset, $limit);

    // Apply status filter.
    if ($status === 'active') {
      $query->condition('used', 0)
        ->condition('expires', $current_time, '>=');
    }
    elseif ($status === 'expired') {
      $query->condition('used', 0)
        ->condition('expires', $current_time, '<');
    }
    elseif ($status === 'used') {
      $query->condition('used', 1);
    }

    $results = $query->execute()->fetchAll();

    // Format results.
    $links = [];
    foreach ($results as $link) {
      $status_value = 'active';
      if ($link->used) {
        $status_value = 'used';
      }
      elseif ($link->expires < $current_time) {
        $status_value = 'expired';
      }

      $link_data = [
        'id' => (int) $link->id,
        'hash' => $link->hash,
        'status' => $status_value,
        'created' => (int) $link->created,
        'created_human' => date('Y-m-d H:i:s', $link->created),
        'expires' => (int) $link->expires,
        'expires_human' => date('Y-m-d H:i:s', $link->expires),
      ];

      if ($link->used) {
        $link_data['used_at'] = (int) $link->used_at;
        $link_data['used_at_human'] = date('Y-m-d H:i:s', $link->used_at);
        $link_data['used_by_ip'] = $link->used_by_ip ?: NULL;
      }

      $links[] = $link_data;
    }

    // Get total count.
    $count_query = $this->database->select('onetimelogin_urls', 'o');
    if ($status === 'active') {
      $count_query->condition('used', 0)
        ->condition('expires', $current_time, '>=');
    }
    elseif ($status === 'expired') {
      $count_query->condition('used', 0)
        ->condition('expires', $current_time, '<');
    }
    elseif ($status === 'used') {
      $count_query->condition('used', 1);
    }
    $total = (int) $count_query->countQuery()->execute()->fetchField();

    // Build response.
    $response_data = [
      'success' => TRUE,
      'data' => [
        'links' => $links,
        'pagination' => [
          'total' => $total,
          'limit' => $limit,
          'offset' => $offset,
          'count' => count($links),
        ],
      ],
    ];

    return new ResourceResponse($response_data, 200);
  }

}
