<?php

declare(strict_types=1);

namespace Drupal\onetimelogin\Plugin\rest\resource;

use Drupal\Core\Session\AccountProxyInterface;
use Drupal\onetimelogin\Service\ShortUrlService;
use Drupal\rest\Plugin\ResourceBase;
use Drupal\rest\ResourceResponse;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Provides a resource to revoke one-time login links.
 *
 * @RestResource(
 *   id = "onetimelogin_revoke",
 *   label = @Translation("Revoke One-Time Login Link"),
 *   uri_paths = {
 *     "canonical" = "/api/v1/onetimelogin/revoke/{hash}"
 *   }
 * )
 */
class RevokeResource extends ResourceBase {

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The short URL service.
   *
   * @var \Drupal\onetimelogin\Service\ShortUrlService
   */
  protected $shortUrlService;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Constructs a RevokeResource object.
   *
   * @param array $configuration
   *   Configuration array.
   * @param string $plugin_id
   *   Plugin ID.
   * @param mixed $plugin_definition
   *   Plugin definition.
   * @param array $serializer_formats
   *   Serializer formats.
   * @param \Psr\Log\LoggerInterface $logger
   *   Logger service.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   Current user.
   * @param \Drupal\onetimelogin\Service\ShortUrlService $short_url_service
   *   Short URL service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   Request stack.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    array $serializer_formats,
    LoggerInterface $logger,
    AccountProxyInterface $current_user,
    ShortUrlService $short_url_service,
    RequestStack $request_stack,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $serializer_formats, $logger);
    $this->currentUser = $current_user;
    $this->shortUrlService = $short_url_service;
    $this->requestStack = $request_stack;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->getParameter('serializer.formats'),
      $container->get('logger.factory')->get('onetimelogin'),
      $container->get('current_user'),
      $container->get('onetimelogin.short_url_service'),
      $container->get('request_stack')
    );
  }

  /**
   * Responds to DELETE requests.
   *
   * @param string $hash
   *   The hash to revoke.
   *
   * @return \Drupal\rest\ResourceResponse
   *   The response.
   */
  public function delete(string $hash): ResourceResponse {
    // Check permission.
    if (!$this->currentUser->hasPermission('revoke one-time login')) {
      throw new AccessDeniedHttpException('Insufficient permissions to revoke one-time login links.');
    }

    // Get client IP.
    $request = $this->requestStack->getCurrentRequest();
    $ip_address = $request ? $request->getClientIp() : NULL;

    // Attempt to revoke the link.
    $revoked = $this->shortUrlService->revokeLink(
      $hash,
      (int) $this->currentUser->id(),
      $ip_address
    );

    if (!$revoked) {
      throw new NotFoundHttpException('Link not found or already used.');
    }

    // Return success response.
    $response_data = [
      'success' => TRUE,
      'data' => [
        'hash' => $hash,
        'revoked_by' => [
          'uid' => (int) $this->currentUser->id(),
          'username' => $this->currentUser->getAccountName(),
        ],
        'revoked_at' => time(),
        'revoked_at_human' => date('Y-m-d H:i:s'),
      ],
      'message' => 'Link revoked successfully.',
    ];

    return new ResourceResponse($response_data, 200);
  }

}
