<?php

declare(strict_types=1);

namespace Drupal\onetimelogin\Plugin\rest\resource;

use Drupal\Core\Database\Connection;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\rest\Plugin\ResourceBase;
use Drupal\rest\ResourceResponse;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;

/**
 * Provides a resource for one-time login statistics.
 *
 * @RestResource(
 *   id = "onetimelogin_statistics",
 *   label = @Translation("One-Time Login Statistics"),
 *   uri_paths = {
 *     "canonical" = "/api/v1/onetimelogin/statistics"
 *   }
 * )
 */
class StatisticsResource extends ResourceBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * Constructs a StatisticsResource object.
   *
   * @param array $configuration
   *   Configuration array.
   * @param string $plugin_id
   *   Plugin ID.
   * @param mixed $plugin_definition
   *   Plugin definition.
   * @param array $serializer_formats
   *   Serializer formats.
   * @param \Psr\Log\LoggerInterface $logger
   *   Logger service.
   * @param \Drupal\Core\Database\Connection $database
   *   Database connection.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   Current user.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   Time service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    array $serializer_formats,
    LoggerInterface $logger,
    Connection $database,
    AccountProxyInterface $current_user,
    TimeInterface $time,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $serializer_formats, $logger);
    $this->database = $database;
    $this->currentUser = $current_user;
    $this->time = $time;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->getParameter('serializer.formats'),
      $container->get('logger.factory')->get('onetimelogin'),
      $container->get('database'),
      $container->get('current_user'),
      $container->get('datetime.time')
    );
  }

  /**
   * Responds to GET requests.
   *
   * @return \Drupal\rest\ResourceResponse
   *   The response.
   */
  public function get(): ResourceResponse {
    // Check permission.
    if (!$this->currentUser->hasPermission('view one-time login statistics')) {
      throw new AccessDeniedHttpException('Insufficient permissions to view statistics.');
    }

    $current_time = $this->time->getRequestTime();

    // Overall statistics.
    $total_generated = (int) $this->database->select('onetimelogin_urls', 'o')
      ->countQuery()
      ->execute()
      ->fetchField();

    $total_used = (int) $this->database->select('onetimelogin_urls', 'o')
      ->condition('used', 1)
      ->countQuery()
      ->execute()
      ->fetchField();

    $total_expired = (int) $this->database->select('onetimelogin_urls', 'o')
      ->condition('used', 0)
      ->condition('expires', $current_time, '<')
      ->countQuery()
      ->execute()
      ->fetchField();

    $total_active = (int) $this->database->select('onetimelogin_urls', 'o')
      ->condition('used', 0)
      ->condition('expires', $current_time, '>=')
      ->countQuery()
      ->execute()
      ->fetchField();

    // Today's statistics.
    $today_start = strtotime('today');

    $today_generated = (int) $this->database->select('onetimelogin_urls', 'o')
      ->condition('created', $today_start, '>=')
      ->countQuery()
      ->execute()
      ->fetchField();

    $today_used = (int) $this->database->select('onetimelogin_urls', 'o')
      ->condition('used', 1)
      ->condition('used_at', $today_start, '>=')
      ->countQuery()
      ->execute()
      ->fetchField();

    // Last 7 days.
    $week_ago = $current_time - (7 * 24 * 60 * 60);

    $week_generated = (int) $this->database->select('onetimelogin_urls', 'o')
      ->condition('created', $week_ago, '>=')
      ->countQuery()
      ->execute()
      ->fetchField();

    $week_used = (int) $this->database->select('onetimelogin_urls', 'o')
      ->condition('used', 1)
      ->condition('used_at', $week_ago, '>=')
      ->countQuery()
      ->execute()
      ->fetchField();

    // Calculate usage rate.
    $usage_rate = $total_generated > 0
      ? round(($total_used / $total_generated) * 100, 2)
      : 0;

    // Build response.
    $response_data = [
      'success' => TRUE,
      'data' => [
        'overall' => [
          'total_generated' => $total_generated,
          'total_used' => $total_used,
          'total_active' => $total_active,
          'total_expired' => $total_expired,
          'usage_rate' => $usage_rate,
        ],
        'today' => [
          'generated' => $today_generated,
          'used' => $today_used,
        ],
        'last_7_days' => [
          'generated' => $week_generated,
          'used' => $week_used,
        ],
        'generated_at' => $current_time,
        'generated_at_human' => date('Y-m-d H:i:s', $current_time),
      ],
    ];

    return new ResourceResponse($response_data, 200);
  }

}
