<?php

namespace Drupal\onetimelogin\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Provides short URL generation and management for one-time login.
 *
 * Generates cryptographically secure short URL hashes that map to full paths.
 * Each hash is stored in the database with an expiration time of 24 hours.
 */
class ShortUrlService {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $database;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected LoggerChannelFactoryInterface $loggerFactory;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   Database service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   Logger factory service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   Config factory service.
   */
  public function __construct(
    Connection $database,
    LoggerChannelFactoryInterface $loggerFactory,
    ConfigFactoryInterface $configFactory,
  ) {
    $this->database = $database;
    $this->loggerFactory = $loggerFactory;
    $this->configFactory = $configFactory;
  }

  /**
   * Generate a secure short URL hash for a path.
   *
   * Uses cryptographically secure random bytes instead of MD5 hashing.
   * Each hash is stored in the database with a 24-hour expiration.
   * Includes collision detection with retry logic (max 5 attempts).
   *
   * @param string $path
   *   The full path to shorten (e.g., '/user/20/one-time-login').
   *
   * @return string
   *   The short URL hash (e.g., '26b6e75d3a9f').
   *
   * @throws \InvalidArgumentException
   *   If path is empty or invalid.
   * @throws \RuntimeException
   *   If unable to generate unique hash after max attempts.
   * @throws \Exception
   *   If database insertion fails.
   */
  public function generateShortUrl(string $path): string {
    if (empty($path)) {
      throw new \InvalidArgumentException('Path cannot be empty.');
    }

    // Ensure path starts with forward slash for consistency.
    if (!str_starts_with($path, '/')) {
      $path = '/' . $path;
    }

    // Try to generate a unique hash with collision detection.
    $max_attempts = 5;
    $attempt = 0;

    while ($attempt < $max_attempts) {
      // Generate cryptographically secure hash (12 hex chars = 48 bits).
      // 12 characters provides 281 trillion combinations (16^12).
      $hash = bin2hex(random_bytes(6));

      // Check if hash already exists.
      $exists = $this->database->select('onetimelogin_urls', 'o')
        ->fields('o', ['id'])
        ->condition('hash', $hash)
        ->execute()
        ->fetchField();

      if (!$exists) {
        // Hash is unique, proceed with insertion.
        // Get configured expiration time (default 24 hours).
        $config = $this->configFactory->get('onetimelogin.settings');
        $expiration_seconds = $config->get('link_expiration') ?? 86400;

        try {
          $this->database->insert('onetimelogin_urls')
            ->fields(
                    [
                      'hash' => $hash,
                      'path' => $path,
                      'created' => time(),
                      'expires' => time() + $expiration_seconds,
                    ]
                )
            ->execute();

          $this->loggerFactory->get('onetimelogin')->debug(
                'Generated short URL: @hash -> @path (attempt @attempt)',
                ['@hash' => $hash, '@path' => $path, '@attempt' => $attempt + 1]
            );

          return $hash;
        }
        catch (\Exception $e) {
          $this->loggerFactory->get('onetimelogin')->error(
                'Error generating short URL: @msg',
                ['@msg' => $e->getMessage()]
            );
          throw $e;
        }
      }

      // Collision detected, log and retry.
      $this->loggerFactory->get('onetimelogin')->warning(
            'Hash collision detected: @hash (attempt @attempt of @max)',
            ['@hash' => $hash, '@attempt' => $attempt + 1, '@max' => $max_attempts]
        );
      $attempt++;
    }

    // Failed to generate unique hash after max attempts.
    $this->loggerFactory->get('onetimelogin')->error(
          'Failed to generate unique hash after @max attempts for path: @path',
          ['@max' => $max_attempts, '@path' => $path]
      );
    throw new \RuntimeException('Failed to generate unique short URL hash after maximum attempts.');
  }

  /**
   * Retrieve the path for a given hash and mark as used.
   *
   * Validates the hash format, checks expiration, and ensures single-use.
   * Expired hashes are automatically deleted from the database.
   *
   * @param string $hash
   *   The short URL hash.
   * @param string|null $ip_address
   *   IP address of the user accessing the link.
   *
   * @return string|null
   *   The full path if valid and not expired, NULL otherwise.
   */
  public function getPath(string $hash, ?string $ip_address = NULL): ?string {
    if (!preg_match('/^[a-f0-9]{8}$/', $hash)) {
      $this->loggerFactory->get('onetimelogin')->warning(
            'Invalid hash format: @hash',
            ['@hash' => $hash]
        );
      return NULL;
    }

    $result = $this->database->select('onetimelogin_urls', 'o')
      ->fields('o', ['path', 'expires', 'used'])
      ->condition('hash', $hash)
      ->execute()
      ->fetch();

    if (!$result) {
      $this->loggerFactory->get('onetimelogin')->warning(
            'Short URL hash not found: @hash',
            ['@hash' => $hash]
        );
      return NULL;
    }

    // Check if already used.
    if ($result->used) {
      $this->loggerFactory->get('onetimelogin')->warning(
            'Short URL hash already used: @hash',
            ['@hash' => $hash]
        );
      return NULL;
    }

    // Check if expired.
    if ($result->expires < time()) {
      $this->database->delete('onetimelogin_urls')
        ->condition('hash', $hash)
        ->execute();

      $this->loggerFactory->get('onetimelogin')->warning(
            'Short URL hash expired: @hash',
            ['@hash' => $hash]
        );
      return NULL;
    }

    // Mark as used.
    $this->markAsUsed($hash, $ip_address);

    return $result->path;
  }

  /**
   * Mark a hash as used.
   *
   * @param string $hash
   *   The short URL hash.
   * @param string|null $ip_address
   *   IP address of the user.
   */
  public function markAsUsed(string $hash, ?string $ip_address = NULL): void {
    $this->database->update('onetimelogin_urls')
      ->fields(
              [
                'used' => 1,
                'used_at' => time(),
                'used_by_ip' => $ip_address,
              ]
          )
      ->condition('hash', $hash)
      ->execute();

    $this->loggerFactory->get('onetimelogin')->info(
          'One-time login link used: @hash from IP @ip',
          ['@hash' => $hash, '@ip' => $ip_address ?? 'unknown']
      );
  }

  /**
   * Revoke a one-time login link.
   *
   * @param string $hash
   *   The short URL hash.
   * @param int $admin_uid
   *   The UID of the admin revoking the link.
   * @param string|null $ip_address
   *   IP address of the admin.
   *
   * @return bool
   *   TRUE if revoked successfully, FALSE if link not found or already used.
   */
  public function revokeLink(string $hash, int $admin_uid, ?string $ip_address = NULL): bool {
    // Check if link exists and is not already used.
    $link = $this->database->select('onetimelogin_urls', 'o')
      ->fields('o', ['used'])
      ->condition('hash', $hash)
      ->execute()
      ->fetchObject();

    if (!$link) {
      return FALSE;
    }

    if ($link->used) {
      return FALSE;
    }

    // Mark as used to prevent further usage.
    $this->database->update('onetimelogin_urls')
      ->fields(
              [
                'used' => 1,
                'used_at' => time(),
                'used_by_ip' => $ip_address,
              ]
          )
      ->condition('hash', $hash)
      ->execute();

    $this->loggerFactory->get('onetimelogin')->notice(
          'One-time login link revoked: @hash by admin UID @admin_uid from IP @ip',
          ['@hash' => $hash, '@admin_uid' => $admin_uid, '@ip' => $ip_address ?? 'unknown']
      );

    return TRUE;
  }

  /**
   * Delete a hash after use (one-time login).
   *
   * @param string $hash
   *   The short URL hash.
   */
  public function deleteHash(string $hash): void {
    $this->database->delete('onetimelogin_urls')
      ->condition('hash', $hash)
      ->execute();

    $this->loggerFactory->get('onetimelogin')->debug(
          'Deleted one-time login hash: @hash',
          ['@hash' => $hash]
      );
  }

}
