<?php

namespace Drupal\onlyoffice_form\Plugin\Field\FieldType;

/**
 * Copyright (c) Ascensio System SIA 2025.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

use Drupal\Core\Field\FieldItemBase;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\DataDefinition;

/**
 * Plugin implementation of the 'onlyoffice_form' field type.
 *
 * @FieldType(
 *   id = "onlyoffice_form",
 *   label = @Translation("ONLYOFFICE form"),
 *   description = @Translation("This field stores a reference to an ONLYOFFICE form media entity with optional description."),
 *   category = @Translation("ONLYOFFICE"),
 *   default_widget = "onlyoffice_form_widget",
 *   default_formatter = "onlyoffice_form_formatter",
 * )
 */
class OnlyofficeFormItem extends FieldItemBase {

  /**
   * {@inheritdoc}
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition) {
    // Define the field properties.
    $properties['target_id'] = DataDefinition::create('integer')
      ->setLabel(new TranslatableMarkup('Media ID'))
      ->setDescription(new TranslatableMarkup('The ID of the ONLYOFFICE form media entity.'))
      ->setRequired(TRUE);

    $properties['description'] = DataDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Description'))
      ->setDescription(new TranslatableMarkup('A description of the form.'));

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public static function schema(FieldStorageDefinitionInterface $field_definition) {
    return [
      'columns' => [
        'target_id' => [
          'type' => 'int',
          'unsigned' => TRUE,
          'not null' => TRUE,
        ],
        'description' => [
          'type' => 'varchar',
          'length' => 255,
          'not null' => FALSE,
        ],
      ],
      'indexes' => [
        'target_id' => ['target_id'],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function isEmpty() {
    $target_id = $this->get('target_id')->getValue();
    return $target_id === NULL || $target_id === 0;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultFieldSettings() {
    return [
      'description_field' => TRUE,
    ] + parent::defaultFieldSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function fieldSettingsForm(array $form, FormStateInterface $form_state) {
    $element = parent::fieldSettingsForm($form, $form_state);

    $element['description_field'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable description field'),
      '#default_value' => $this->getSetting('description_field'),
      '#description' => $this->t('The description field allows users to enter a description about the form.'),
      '#weight' => 10,
    ];

    return $element;
  }

}
