<?php

namespace Drupal\onomasticon\Plugin\Filter;

use Drupal\Component\Utility\Html;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\filter\FilterProcessResult;
use Drupal\filter\Plugin\FilterBase;
use Drupal\taxonomy\Entity\Vocabulary;
use Drupal\taxonomy\Entity\Term;

/**
 * @Filter(
 *   id = "filter_onomasticon",
 *   title = @Translation("Onomasticon Filter"),
 *   description = @Translation("Adds glossary information to words."),
 *   type = Drupal\filter\Plugin\FilterInterface::TYPE_TRANSFORM_IRREVERSIBLE,
 *   settings = {
 *     "onomasticon_vocabulary" = {},
 *     "onomasticon_definition_field" = "description",
 *     "onomasticon_definition_filters" = false,
 *     "onomasticon_tag" = "dfn",
 *     "onomasticon_disabled" = "abbr audio button cite code dfn form meta object pre style script video",
 *     "onomasticon_implement" = "extra_element",
 *     "onomasticon_orientation" = "below",
 *     "onomasticon_cursor" = "default",
 *     "onomasticon_repetition" = "",
 *     "onomasticon_ignorecase" = false,
 *     "onomasticon_termlink" = false,
 *     "onomasticon_showsynonym" = false,
 *   },
 * )
 */
class FilterOnomasticon extends FilterBase
{

  /**
   * @var \DOMDocument
   * This var is the base element all replacements are made on.
   */
  private $htmlDom;

  /**
   * @var array
   * Contains all paths of processed DOM Nodes to avoid
   * duplicate replacements.
   */
  private $processedPaths = [];

  /**
   * @var array
   * Collection of replacements. Gets applied to $htmlDom
   * one the complete tree has been traversed.
   */
  private $htmlReplacements = [];

  /**
   * @var Term[]
   * Simple cache mechanism for loaded terms. Also works
   * as a list of already processed terms.
   */
  private $terms = [];

  /**
   * @var array
   *
   * Simple cache mechanism for loaded terms. Also works
   * as a list of already processed terms.
   * 
   */
  private $termCache = [];

  /**
   * @var array
   *
   */
  private static $compiledTerms = [];

  /**
   * Main filter function as expected by Drupal.
   *
   * @param string $text
   *   Text to be processed.
   * @param string $langcode
   *   Language code.
   * 
   * @return \Drupal\filter\FilterProcessResult|string
   */
  public function process($text, $langcode): FilterProcessResult|string {
    // Check if a vocabulary has been set.
    if (empty($this->settings['onomasticon_vocabulary'])) {
      return new FilterProcessResult($text);
    }

    // Use DOMDocument with libxml error handling
    libxml_use_internal_errors(true);
    $dom = new \DOMDocument('1.0', 'UTF-8');
    $dom->strictErrorChecking = false;
    $dom->recover = true;

    // Wrap text in body tags to ensure proper parsing
    $wrappedText = '<html><body>' . $text . '</body></html>';

    // Attempt to load HTML with error suppression
    $loadSuccess = $dom->loadHTML(
      $wrappedText,
      LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD | LIBXML_NOERROR | LIBXML_NOWARNING
    );

    // Check for loading errors
    if (!$loadSuccess) {
      $errors = libxml_get_errors();
      libxml_clear_errors();

      // Log errors if needed
      if (!empty($errors)) {
        \Drupal::logger('onomasticon_filter')->warning('HTML parsing errors: @errors', [
          '@errors' => implode(', ', array_map(function ($error) {
            return $error->message;
          }, $errors))
        ]);
      }

      // Fallback to original text if parsing fails
      return new FilterProcessResult($text);
    }

    // Get the body element
    $body = $dom->getElementsByTagName('body')->item(0);

    // Reset replacements array
    $this->htmlReplacements = [];
    $this->processedPaths = [];

    // Walk the DOM and replace terms with definitions
    $this->processChildren($body);

    foreach ($this->htmlReplacements as $r) {
      try {
        // Create a new document fragment
        $fragment = $dom->createDocumentFragment();

        // Normalize HTML entities
        $normalizedHtml = $this->htmlEntitiesNormalizeXml($r['html']);

        // Safely load the fragment
        $fragmentLoadSuccess = $fragment->appendXML($normalizedHtml);

        if ($fragmentLoadSuccess) {
          // Replace the node
          $r['dom']->parentNode->replaceChild($fragment, $r['dom']);
        }
      } catch (\Exception $e) {
        // Log any replacement errors
        \Drupal::logger('onomasticon_filter')->error(
          'Error during HTML replacement: @error',
          ['@error' => $e->getMessage()]
        );
      }
    }

    // Export all "body" childes as HTML 5.
    $text = '';
    foreach ($body->childNodes as $childNode) {
      $text .= $dom->saveHTML($childNode);
    }
    // Prepare return object.
    $result = new FilterProcessResult($text);
    // We force the cache to be cleared automatically on changes to the vocabulary.
    foreach ($this->settings['onomasticon_vocabulary'] as $vocabulary) {
      $result->addCacheTags([
        'taxonomy_term_list:' . $vocabulary,
      ]);
    }

    // Remove custom tag if needed.
    $remove_onomasticon_tag = $this->removeCustomTag('nonomasticon', $text);
    if ($remove_onomasticon_tag) {
      $result->setProcessedText($remove_onomasticon_tag);
    }

    return $result;
  }

  /**
   * Remove custom tags added for editing purposes.
   *
   * @param string $tag
   *   The custom html tag.
   * @param string $text
   *   The text where should replace tag.
   *
   * @return string
   *   A processed text without given html tag.
   */
  private function removeCustomTag(string $tag, string $text): string {
    return preg_replace('/<\/?' . $tag . '[^>]*>/', '', $text);
  }

  /**
   * Unicode-proof htmlentities function.
   * Returns 'normal' chars as chars and special characters
   * as numeric html entities.
   *
   * @param string $string
   * @return string
   * */
  protected function htmlEntitiesNormalizeXml($string): string {

    // Get rid of existing entities and double-escape.
    $string = html_entity_decode(stripslashes($string), ENT_QUOTES, 'UTF-8');
    $result = '';
    // Create array of multi-byte characters.
    $ar = preg_split('/(?<!^)(?!$)/u', $string);
    foreach ($ar as $c) {
      $o = ord($c);
      if (
        // Any multi-byte character's length is > 1
        (strlen($c) > 1)
        // Control characters are below 32, latin special chars are above 126.
        // Non-latin characters are above 126 as well, including &nbsp;.
        || ($o < 32 || $o > 126)
        // That's the ampersand.
        || ($o == 38)
        /* This following ranges includes single&double quotes,
         * ampersand, hash sign, less- and greater-than signs.
         * We do not want to replace these, otherwise HTML will break,
         * except for the ampersand which is targeted above.
         * Non-breaking spaces &nbsp; are converted to &#160; .
         */
        #|| ($o > 33 && $o < 40) /* quotes + ampersand */
        #|| ($o > 59 && $o < 63) /* html */
      ) {
        // Convert to numeric entity.
        $c = mb_encode_numericentity($c, [0x0, 0xffff, 0, 0xffff], 'UTF-8');
      }
      $result .= $c;
    }
    // Mask ampersands
    $result = str_replace('&#38;', '###amp###', $result);
    $result = html_entity_decode($result, ENT_HTML5);
    $result = str_replace('###amp###', '&#38;', $result);
    // Escape < characters that are not part of tags or comments.
    $result = preg_replace_callback('/<(?!\/?(?:\w+\s*\/?|\!--))/s', function ($matches) {
      return '&lt;';
    }, $result);

    return $this->fixSelfClosingTags($result);
  }

  /**
   * Ensures self-closing tags are properly closed in a large HTML snippet.
   *
   * @param string $html
   *   The HTML content to process.
   *
   * @return string
   *   HTML with self-closing tags fixed.
   */
  public function fixSelfClosingTags(string $html): string {
    // List of common self-closing HTML tags.
    $selfClosingTags = [
      'area','base','br','col','command','embed','hr','img',
      'input','keygen','link','meta','param','source','track','wbr'
    ];

    // Join tags into a regex alternation group
    $tagPattern = implode('|', array_map('preg_quote', $selfClosingTags));

    // Regex explanation:
    // <(tag)      : opening tag with tag name
    // \b          : word boundary
    // (           : capture group for attributes
    //   (?:       : non-capturing group
    //     [^>"\'] : anything except >, " or '
    //     |"[^"]*" : or double-quoted string
    //     |'[^']*' : or single-quoted string
    //   )*
    // )
    // >           : literal closing bracket
    $pattern = '/<(' . $tagPattern . ')\b((?:[^>"\']|"[^"]*"|\'[^\']*\')*)>/i';

    $html = preg_replace_callback($pattern, function($matches) {
      $tag = $matches[1];
      $attrs = rtrim($matches[2]);
      // If already self-closed, leave as is
      if (substr($attrs, -1) === '/') {
        return "<$tag$attrs>";
      }
      return "<$tag$attrs />";
    }, $html);

    return $html;
  }

  /**
   * Traverses the DOM tree (recursive).
   * If current DOMNode element has children,
   * this function calls itself.
   * #text nodes never have any children, there
   * Onomasticon filter is applied.
   *
   * @param $dom \DOMNode
   * @param $tree array
   *
   */
  public function processChildren(\DOMNode $dom, $tree = []) {
    if ($dom->hasChildNodes()) {
      // Children present, this can't be a #text node.
      // Add the tag name to the tree, so we know the hierarchy.
      $tree[] = $dom->nodeName;

      // Create a copy of childNodes to avoid modification during iteration
      $childNodes = iterator_to_array($dom->childNodes);

      // Recursive call on children
      foreach ($childNodes as $child) {
        $this->processChildren($child, $tree);
      }
    } 
    else {
      // No children present => end of tree branch.
      if ($dom->nodeName === '#text' && trim($dom->nodeValue) !== '') {
        // Element is of type #text and has content.

        // First check tree for ancestor tags not allowed.
        $disabledTags = explode(' ', $this->settings['onomasticon_disabled']);

        // Sanitize user input
        $disabledTags = array_map(
          function ($tag) {
            return preg_replace("/[^a-z1-6]*/", "", strtolower(trim($tag)));
          },
          $disabledTags
        );

        // Add Onomasticon tag and anchor tag.
        $disabledTags[] = $this->settings['onomasticon_tag'];
        $disabledTags[] = 'a';
        $disabledTags[] = 'nonomasticon'; // Required for CKEditor plugin.
        $disabledTags = array_unique($disabledTags);

        // Find the bad boys.
        $badTags = array_intersect($disabledTags, $tree);

        if (count($badTags) == 0) {
          // To avoid double replacements, check if this element
          // has been processed already.

          // Note: With DOMDocument, getNodePath() might work differently
          $nodePath = $this->getNodePath($dom);

          if (!in_array($nodePath, $this->processedPaths)) {
            // Original nodeValue (textContent).
            $textOrigin = $dom->nodeValue;

            // Processed text, Onomasticon has been applied.
            $textReplica = $this->replaceTerms($textOrigin);

            // Did the filter find anything?
            if ($textOrigin !== $textReplica) {
              // Indeed, let's save the information for later.
              $this->htmlReplacements[] = [
                'dom' => $dom,
                'html' => $textReplica,
              ];
            }

            // Add element to processed items.
            $this->processedPaths[] = $nodePath;
          }
        }
      }
    }
  }

  /**
   * Generates an XPath-like path for a given DOM node.
   *
   * This path is used to uniquely identify a node within the DOM tree,
   * which helps in preventing duplicate processing of the same node.
   *
   * @param \DOMNode $node
   *   The DOM node for which to generate the path.
   *
   * @return string
   *   The XPath-like path for the node, or an empty string if the node type
   *   is not supported.
   */
  protected function getNodePath(\DOMNode $node): string {
    if ($node->nodeType === XML_TEXT_NODE) {
      $parent = $node->parentNode;
      $index = 0;
      foreach ($parent->childNodes as $childNode) {
        if ($childNode === $node) {
          break;
        }
        $index++;
      }
      return $this->getNodePath($parent) . "/text()[" . ($index + 1) . "]";
    }

    if ($node->nodeType === XML_ELEMENT_NODE) {
      $parent = $node->parentNode;
      if (!$parent) {
        return $node->nodeName;
      }

      $index = 0;
      foreach ($parent->childNodes as $childNode) {
        if ($childNode === $node) {
          break;
        }
        if ($childNode->nodeName === $node->nodeName) {
          $index++;
        }
      }

      return $this->getNodePath($parent) . '/' . $node->nodeName . '[' . ($index + 1) . ']';
    }

    return '';
  }

  /**
   * This is the actual filter function.
   *
   * @param string $text 
   *   String containing a #text DOMNode value.
   * 
   * @return string 
   *   Processed string.
   */
  public function replaceTerms($text) {
    $pregLimit = $this->settings['onomasticon_repetition'] ? 1 : -1;
    $replacements = [];

    foreach ($this->getCompiledTerms() as $index => $termData) {
      $cacheKey = $termData['term_id'] . ':' . $termData['index'];

      if ($this->settings['onomasticon_repetition'] && array_key_exists($cacheKey, $this->getCacheTerms())) {
        continue;
      }

      // Use precomputed lowercase text for case-insensitive search
      if ($this->settings['onomasticon_ignorecase']) {
        $pos = strpos(strtolower($text), $termData['name_lower']);
      } else {
        $pos = strpos($text, $termData['name']);
        if ($pos === FALSE) {
          $pos = strpos($text, ucfirst($termData['name']));
        }
      }

      if ($pos === FALSE) {
        continue;
      }
      // Set the correct cased needle.
      $needle = substr($text, $pos, $termData['length']);
      $needles = [$needle];
      if ($this->settings['onomasticon_ignorecase']) {
        if ($needle === strtolower($needle)) {
          $needles[] = ucfirst($needle);
        } 
        else {
          $needles[] = strtolower($needle);
        }
      }
      $this->addCacheTerm($cacheKey);
      $description = $termData['description'];

      // Set the implementation method.
      $implement = $this->settings['onomasticon_implement'];
      if ($implement == 'attr_title') {
        $description = strip_tags($description);
        // Title attribute is enclosed in double quotes,
        // we need to escape double quotes in description.
        // TODO: Instead of removing double quotes altogether, escape them.
        $description = str_replace('"', '', $description);
        // Replace no-breaking spaces with normal ones.
        $description = str_replace('&nbsp;', ' ', $description);
        // Trim multiple white-space characters with single space.
        $description = preg_replace('/\s+/m', ' ', $description);
      }

      foreach ($needles as $n => $currentNeedle) {
        $onomasticon = [
          '#theme' => 'onomasticon',
          '#tag' => $this->settings['onomasticon_tag'],
          '#needle' => $currentNeedle . $termData['suffix'],
          '#description' => $description,
          '#implement' => $implement,
          '#orientation' => $this->settings['onomasticon_orientation'],
          '#cursor' => $this->settings['onomasticon_cursor'],
          '#termlink' => $this->settings['onomasticon_termlink'],
          '#termpath' => $termData['term']->toUrl()->toString(),
          '#term' => $termData['term'],
        ];

        $placeholder = '###' . $cacheKey . ($n === 1 ? '-alt' : '') . '###';
        $replacements[$placeholder] = trim(\Drupal::service('renderer')->render($onomasticon));

        $count = NULL;
        $text = preg_replace(
          "/(?<![a-zA-Z0-9_äöüÄÖÜ])" . preg_quote($currentNeedle, '/') . "(?![a-zA-Z0-9_äöüÄÖÜ])/",
          $placeholder,
          $text,
          $pregLimit,
          $count
        );
        if ($count === 0) {
          unset($this->termCache[$cacheKey]);
        }
      }
    }

    foreach ($replacements as $placeholder => $replacement) {
      $text = str_replace($placeholder, $replacement, $text, $pregLimit);
    }

    return $text;
  }

  /**
   * Retrieves and compiles taxonomy terms for filtering.
   *
   * This function fetches terms from the configured vocabularies, including their
   * translations and synonyms. It pre-calculates and caches the data needed for
   * the replacement process to optimize performance. The terms are sorted by
   * length in descending order to ensure longer terms are matched before
   * shorter ones.
   *
   * @return array
   *   An array of compiled term data, sorted by term length. Each item in the
   *   array is an associative array with the following keys:
   *   - 'term_id': The taxonomy term ID.
   *   - 'index': The index of the name (0 for the main term, >0 for synonyms).
   *   - 'name': The term name or synonym.
   *   - 'suffix': A suffix to append (e.g., the original term name for a synonym).
   *   - 'name_lower': The lowercased version of the term name.
   *   - 'description': The pre-processed term description.
   *   - 'term': The translated taxonomy term entity.
   *   - 'length': The string length of the term name.
   */
  private function getCompiledTerms() {
    $vocabulary_id = implode(':', $this->settings['onomasticon_vocabulary']);
    $language = \Drupal::languageManager()
      ->getCurrentLanguage(LanguageInterface::TYPE_CONTENT)
      ->getId();
    $cacheKey = $language . '::' . $vocabulary_id;

    if (isset(self::$compiledTerms[$cacheKey])) {
      return self::$compiledTerms[$cacheKey];
    }

    $compiledData = [];

    foreach ($this->getTaxonomyTerms() as $term) {
      if (!$term->hasTranslation($language)) {
        continue;
      }

      $translatedTerm = $term->getTranslation($language);
      $termNames = [$translatedTerm->label()];

      // Add synonyms if available
      if (!empty(\Drupal::hasService('synonyms.provider_service'))) {
        $synonyms = \Drupal::service('synonyms.provider_service')->getEntitySynonyms($translatedTerm);
        $termNames = array_merge($termNames, $synonyms);
      }

      // Pre-calculate description
      $customDefinitionField = $this->settings['onomasticon_definition_field'];
      $fieldName = $translatedTerm->hasField($customDefinitionField) ? $customDefinitionField : 'description';
      $description = $translatedTerm->$fieldName->value ?? '';

      $fieldType = $translatedTerm->get($fieldName)->getFieldDefinition()->getType();
      if ($this->settings['onomasticon_definition_filters'] && in_array($fieldType, ['text', 'text_with_summary', 'text_long'])) {
        $filterFormatStorage = \Drupal::entityTypeManager()->getStorage('filter_format');
        /** @var \Drupal\filter\FilterFormatInterface $filterFormat */
        $filterFormat = $filterFormatStorage->load($translatedTerm->get($fieldName)->format);

        if ($filterFormat && !$filterFormat->filters('filter_onomasticon')->status) {
          /** @var \Drupal\filter\FilterProcessService $filterProcessor */
          $description = (string) $translatedTerm->$fieldName->processed;
        }
      }

      $showsynonym = $this->settings['onomasticon_showsynonym'];
      foreach ($termNames as $index => $termName) {
        $compiledData[] = [
          'term_id' => $term->id(),
          'index' => $index,
          'name' => $termName,
          'suffix' => ($showsynonym && $index > 0 ? ' (' . $termNames[0] . ')' : ''),
          'name_lower' => strtolower($termName),
          'description' => $description,
          'term' => $translatedTerm,
          'length' => strlen($termName),
        ];
      }
    }

    // Sort by length (longest first) for better matching
    usort($compiledData, function ($a, $b) {
      return $b['length'] - $a['length'];
    });

    self::$compiledTerms[$cacheKey] = $compiledData;
    return $compiledData;
  }

  /**
   * Get terms id already processed from cache.
   */
  private function getCacheTerms() {
    // We choose between the local cache and the global cache.
    return $this->settings['onomasticon_repetition'] === 'page' ?
      onomasticon_get_term_cache() :
      $this->termCache;
  }

  /**
   * Register a term id in the cache.
   */
  private function addCacheTerm($tid) {
    // We choose between the local cache and the global cache.
    $this->settings['onomasticon_repetition'] === 'page' ?
      onomasticon_set_term_cache($tid) :
      $this->termCache[$tid] = TRUE;
  }

  /**
   * Singleton to retrieve all terms.
   *
   * @return Term[]
   */
  private function getTaxonomyTerms() {
    $taxonomyTermStorage = \Drupal::entityTypeManager()
      ->getStorage('taxonomy_term');
    $term_ids = $taxonomyTermStorage->getQuery()
      ->condition('vid', $this->settings['onomasticon_vocabulary'], 'IN')
      ->condition('status', TRUE)
      ->accessCheck()
      ->execute();
    if (!empty($term_ids)) {
      $terms = $taxonomyTermStorage->loadMultiple($term_ids);
    }
    \Drupal::moduleHandler()->alter('onomasticon_terms', $terms);
    return $terms;
  }

  /**
   * Filter settings form.
   *
   * @param array $form
   *  Form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *  Form state.
   * 
   * @return array
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
    $vocabularies = Vocabulary::loadMultiple();
    $options = [];
    foreach ($vocabularies as $vocabulary) {
      $options[$vocabulary->id()] = $vocabulary->get('name');
    }
    $form['onomasticon_vocabulary'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Vocabularies'),
      '#options' => $options,
      '#default_value' => $this->settings['onomasticon_vocabulary'],
      '#description' => $this->t('Choose the vocabularies that hold the glossary terms.'),
      '#element_validate' => [[static::class, 'validateOptions']],
    ];
    $form['onomasticon_definition_field'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Definition'),
      '#default_value' => $this->settings['onomasticon_definition_field'],
      '#description' => $this->t('Enter the machine field name that holds the definition of the word. Ignore if you put your definitions in the default description taxonomy field.'),
    ];
    $form['onomasticon_definition_filters'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Run text filters on term description'),
      '#default_value' => $this->settings['onomasticon_definition_filters'],
      '#description' => $this->t('If checked and the selected description source is a formatted text field, all text filters for the format used will run on the description value. <br /><i><b>Caution:</b> This can lead to infinite loops and break your site if term descriptions contain other glossary terms.</i>'),
    ];
    $form['onomasticon_tag'] = [
      '#type' => 'select',
      '#title' => $this->t('HTML tag'),
      '#options' => [
        'dfn' => $this->t('Definition (dfn)'),
        'abbr' => $this->t('Abbreviation (abbr)'),
        'cite' => $this->t('Title of work (cite)'),
      ],
      '#default_value' => $this->settings['onomasticon_tag'],
      '#description' => $this->t('Choose the HTML tag to contain the glossary term.'),
    ];
    $form['onomasticon_disabled'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Disabled tags'),
      '#default_value' => $this->settings['onomasticon_disabled'],
      '#description' => $this->t('Enter all HTML elements in which terms should not be replaced. Anchor tag as well as the default HTML tag are added to that list automatically.'),
    ];
    $form['onomasticon_implement'] = [
      '#type' => 'select',
      '#title' => $this->t('Implementation'),
      '#options' => [
        'extra_element' => $this->t('Extra element'),
        'attr_title' => $this->t('Title attribute'),
        'accessibility_first' => $this->t('Accessibility first'),
      ],
      '#default_value' => $this->settings['onomasticon_implement'],
      '#description' => $this->t('Choose the implementation of the glossary term description. Due to HTML convention, the description will be stripped of any tags as they are not allowed in a tag\'s attribute.'),
    ];
    $form['onomasticon_orientation'] = [
      '#type' => 'select',
      '#title' => $this->t('Orientation'),
      '#options' => [
        'above' => $this->t('Above'),
        'below' => $this->t('Below'),
      ],
      '#default_value' => $this->settings['onomasticon_orientation'],
      '#description' => $this->t('Choose whether the tooltip should appear above or below the hovered glossary term.'),
      '#states' => [
        'visible' => [
          'select[name="filters[filter_onomasticon][settings][onomasticon_implement]"]' => [
            ['value' => 'extra_element'],
            'or',
            ['value' => 'accessibility_first'],
          ],
        ],
      ],
    ];
    $form['onomasticon_cursor'] = [
      '#type' => 'select',
      '#title' => $this->t('Mouse cursor'),
      '#options' => [
        'default' => $this->t('Default (Text cursor)'),
        'help' => $this->t('Help cursor'),
        'none' => $this->t('Hide cursor'),
      ],
      '#default_value' => $this->settings['onomasticon_cursor'],
      '#description' => $this->t('Choose a style the mouse cursor will change to when hovering a glossary term.'),
    ];
    $form['onomasticon_ignorecase'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Ignore case'),
      '#default_value' => $this->settings['onomasticon_ignorecase'],
      '#description' => $this->t('If checked, Onomasticon will find all occurrences of a term regardless of case (even CamelCase will work). If not checked, Onomasticon will only find the exact cased term or with the first letter capitalized (i.e. for start of sentences).'),
    ];
    $form['onomasticon_repetition'] = [
      '#type' => 'select',
      '#title' => $this->t('Repetition'),
      '#options' => [
        '' => $this->t('Add definition to all occurences.'),
        'text' => $this->t('Add definition to first occurrence of term in text, only.'),
        'page' => $this->t('Add definition to first occurence of term in page, only.'),
      ],
      '#default_value' => $this->settings['onomasticon_repetition'],
      '#description' => $this->t('Choose wether to process all occurrences in text, ignore repetition in same text block or ignore repetition in whole page.'),
    ];
    $form['onomasticon_termlink'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Add a link to the term entity.'),
      '#default_value' => $this->settings['onomasticon_termlink'],
      '#description' => $this->t('If you enable this option the tooltip will be extended by a link to the full term entity which enables you to show even more information. This only works if the implementation is done with an extra element.'),
    ];
    $form['onomasticon_showsynonym'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show the original term if a synonym is explained.'),
      '#default_value' => $this->settings['onomasticon_showsynonym'],
      '#description' => $this->t('If you enable this option, any synonym of a term will be followed by the original term in parenthesis.'),
    ];
    return $form;
  }

  /**
   * Form element validation handler.
   * 
   * @param array $element
   *   The allowed_view_modes form element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public static function validateOptions(array &$element, FormStateInterface $form_state): void {
    // Filters the #value property so only selected values appear in the
    // config.
    $value = array_filter($element['#value']);
    $onomasticon_settings = $form_state->getValue(['filters', 'filter_onomasticon']);
    // Validate that vocabulary is set.
    if (empty($value) && $onomasticon_settings['status'] == TRUE) {
      $form_state->setError($element, t('Please select at least one vocabulary.'));
    }
    $form_state->setValueForElement($element, $value);
  }
}
