<?php

namespace Drupal\openagenda\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\openagenda\OpenagendaHelperInterface;
use Drupal\openagenda\OpenagendaConnectorInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use \Drupal;

/**
 * Provides the OpenAgenda geographic filter Block.
 *
 * @Block(
 *   id = "openagenda_geographic_filter_block",
 *   admin_label = @Translation("OpenAgenda - Geographic filter"),
 *   category = @Translation("OpenAgenda"),
 *   context_definitions = {
 *     "node" = @ContextDefinition("entity:node", label = @Translation("Node"))
 *   },
 * )
 */
class OpenagendaGeographicFilterBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The OpenAgenda helper service.
   *
   * @var \Drupal\openagenda\OpenagendaHelperInterface
   */
  protected $helper;

  /**
   * The route match.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * Entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * OpenAgenda module configuration object.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $moduleConfig;

  /**
   * The OpenAgenda connector service.
   *
   * @var \Drupal\openagenda\OpenagendaConnectorInterface
   */
  protected $connector;

  /**
   * {@inheritdoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, RouteMatchInterface $route_match, EntityTypeManagerInterface $entity_type_manager, OpenagendaHelperInterface $helper, OpenagendaConnectorInterface $connector, ConfigFactoryInterface $config_factory) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->routeMatch = $route_match;
    $this->entityTypeManager = $entity_type_manager;
    $this->helper = $helper;
    $this->connector = $connector;
    $this->moduleConfig = $config_factory->get('openagenda.settings');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_route_match'),
      $container->get('entity_type.manager'),
      $container->get('openagenda.helper'),
      $container->get('openagenda.connector'),
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function blockAccess(AccountInterface $account) {
    return AccessResult::allowedIfHasPermission($account, 'access content');
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $node = $this->getContextValue('node');
    $block = [];
    $defaultPlaceholder = $this->moduleConfig->get('openagenda.default_choice_filter_placeholder');
    $placeholder = !empty($this->configuration['input_placeholder']) ? $this->configuration['input_placeholder'] : $this->t($defaultPlaceholder);
    $aria_label = !empty($this->configuration['input_aria_label']) ? $this->configuration['input_aria_label'] : $this->t($defaultPlaceholder);
    $option_tag = !empty($this->configuration['option_tag']) ? $this->configuration['option_tag'] : 'div';
    $filter_type = $this->configuration['filter_type'];
    $options_page_size = !empty($this->configuration['options_page_size']) ? $this->configuration['options_page_size'] : 10;
    $options_selected = NULL;
    // If options limited
    if((bool)$this->configuration['options_limited']){
      // If has options selected
      if(isset($this->configuration['options_selected'][$this->configuration['options_agenda']][$this->configuration['filter_type']])){
        // Get options selected from configuration
        $options_selected_config = $this->configuration['options_selected'][$this->configuration['options_agenda']][$this->configuration['filter_type']] ?? [];
        // Prepare options selected format for filter params
        $options_selected = [];
        foreach($options_selected_config as $k_option => $v_options){
          if((bool)$v_options['enabled']){
            $options_selected[] = [
              'label' => $k_option,
              'value' => $k_option,
            ];
          }
        }

      }
    }

    // Check that we have an OpenAgenda node and that we are hitting the base
    // route (not an event).
    if ($node && $node->hasField('field_openagenda') && $this->routeMatch->getRouteName() == 'entity.node.canonical') {
      $block = [
        '#theme' => 'block__openagenda_geographic_filter',
        '#placeholder' => $placeholder,
        '#aria_label' => $aria_label,
        '#filter_type' => $filter_type,
        '#options_page_size' => $options_page_size,
        '#options_selected' => $options_selected,
        '#option_tag' => $option_tag,
      ];
    }

    return $block;
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state)
  {
    $form = parent::blockForm($form, $form_state);
    $config = $this->getConfiguration();

    $form['input_placeholder'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Placeholder'),
      '#description' => $this->t('Placeholder for the input field.'),
      '#default_value' => isset($config['input_placeholder']) ? $config['input_placeholder'] : $this->moduleConfig->get('openagenda.default_choice_filter_placeholder'),
    ];
    $form['input_aria_label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Aria label'),
      '#description' => $this->t('Aria label for the input field.'),
      '#default_value' => isset($config['input_aria_label']) ? $config['input_aria_label'] : $this->moduleConfig->get('openagenda.default_choice_filter_placeholder'),
    ];
    $form['option_tag'] = [
      '#type' => 'select',
      '#title' => $this->t('Tag for options'),
      '#options' => [
        'div' => $this->t('Default'),
        'label' => $this->t('Label'),
      ],
      '#default_value' => $this->configuration['option_tag'] ?? 'label',
      '#required' => TRUE,
    ];
    $form['filter_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Filter type'),
      '#description' => $this->t('Filter type based on geographic subdivision.'),
      '#options' => [
        'region' => $this->t('Region'),
        'department' => $this->t('Department'),
        'city' => $this->t('City'),
        'adminLevel3' => $this->t('Inter-city'),
        'district' => $this->t('Districts'),
        'locationUid' => $this->t('Location'),
      ],
      '#default_value' => isset($config['filter_type']) ? $config['filter_type'] : NULL,
      '#ajax' => [
        'callback' => [$this, 'ajaxGetFilterOptionsCallback'],
        'event' => 'change',
        // 'event' => 'autocompleteclose',
        'wrapper' => 'settings-options-selected-wrapper',
      ],
    ];

    $form['options_page_size'] = [
      "#type" => "number",
      "#title" => $this->t("Number of displayed options before 'Load more' link"),
      "#min" => 0,
      "#max" => 100,
      '#default_value' => isset($config['options_page_size']) ? $config['options_page_size'] : 10,
    ];

    $form['options_limited'] = [
      "#type" => "checkbox",
      "#title" => $this->t("Select and sort filter values to display"),
      '#default_value' => isset($config['options_limited']) ? $config['options_limited'] : NULL,
    ];

    // Get all nodes of type openagenda
    $nodes_openagenda = $this->entityTypeManager->getStorage('node')->loadByProperties(['type' => 'openagenda']);
    $nodes_options = [];
    foreach ($nodes_openagenda as $node) {
      $nodes_options[$node->id()] = $node->label();
    }

    $form['options_agenda'] = [
      '#type' => 'select',
      '#title' => $this->t('Agenda'),
      '#description' => $this->t('Select the agenda to limit the options.'),
      '#empty_option' => $this->t('- Select an agenda -'),
      '#options' => $nodes_options,
      '#default_value' => isset($config['options_agenda']) ? $config['options_agenda'] : NULL,
      '#states' => [
        'visible' => [
          ':input[name="settings[options_limited]"]' => ['checked' => TRUE],
        ],
      ],
      '#ajax' => [
        'callback' => [$this, 'ajaxGetFilterOptionsCallback'],
        'event' => 'change',
        'wrapper' => 'settings-options-selected-wrapper',
      ],
    ];

    // Create container for options selected
    $form['options_selected_container'] = [
      '#type' => 'container',
      '#attributes' => [
        'id' => 'settings-options-selected-wrapper',
      ],
      '#states' => [
        'visible' => [
          ':input[name="settings[options_limited]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['options_selected_container']['options_selected'] = [
      "#type" => "table",
      '#tree' => TRUE,
      "#header" => [
        // '',
        'label' => [
          'data' => t('Label'),
        ],
        'enabled' => [
          'data' => t('Enabled'),
        ],
        'weight' => [
          'data' => t('Weight'),
        ],
      ],
      "#attributes" => [
        'class' => ['table', 'table-striped'],
      ],
      '#tabledrag' => [[
        'action' => 'order',
        'relationship' => 'sibling',
        'group' => 'draggable-weight',
      ]],
      '#empty' => $this->t('No options available.'),
    ];

    // Get options default selected
    $options_selected_default = $config['options_selected'] ?? [];

    // Get Node Agenda ID default selected
    $agenda_node_id = $config['options_agenda'];
    // Get type filter default selected
    $filter_type = $config['filter_type'];

    // Contains the types of filters supported by OpenAgenda
    $filter_types_aggs = [
      'district' => 'districts',
      'city' => 'cities',
      'department' => 'departments',
      'region' => 'regions',
      'adminLevel3' => 'interCities',
      'locationUid' => 'locations',
    ];

    // Get values submitted after ajax
    $values = $form_state->getUserInput();
    // If has agenda node selected
    if (isset($values['settings']['options_agenda']) && !empty($values['settings']['options_agenda'])) {
      $agenda_node_id = $values['settings']['options_agenda'];
    }
    // If has filter type selected
    if (isset($values['settings']['filter_type']) && !empty($values['settings']['filter_type'])) {
      $filter_type = $values['settings']['filter_type'];
    }

    // If has agenda node selected, filter type is not empty and filter type is supported
    if(!empty($agenda_node_id) && !empty($filter_type) && isset($filter_types_aggs[$filter_type])){
      // Get options selected default for this context
      $options_selected_default = isset($options_selected_default[$agenda_node_id][$filter_type]) ? $options_selected_default[$agenda_node_id][$filter_type] : [];
      // Get filter type request
      $filter_type_aggs = $filter_types_aggs[$filter_type];

      // Get options
      $node = $this->entityTypeManager->getStorage('node')->load($agenda_node_id);
      $agenda_options_getted = $this->connector->getAgendaEvents($node->get('field_openagenda')->uid, [
        'aggsSizeLimit' => 1500,
        'aggs' => [
          [
            'k' => 'items',
            't' => $filter_type_aggs,
            'm' => null,
          ]
        ],
        'size' => 0,
      ]);

      if(!empty($agenda_options_getted)){
        foreach($agenda_options_getted['aggregations']['items'] as $k_item => $v_item){
          $form['options_selected_container']['options_selected'][$v_item['key']] = [
            '#attributes' => ['class' => ['draggable']],
            '#weight' => isset($options_selected_default[$v_item['key']]) ? $options_selected_default[$v_item['key']]['weight'] : $k_item,
            'label' => [
              '#type' => 'markup',
              '#markup' => $v_item['key'],
            ],
            'enabled' => [
              '#type' => 'checkbox',
              '#title' => t('Enabled'),
              '#title_display' => 'invisible',
              '#default_value' => isset($options_selected_default[$v_item['key']]) ? $options_selected_default[$v_item['key']]['enabled'] : false,
              '#value' => isset($options_selected_default[$v_item['key']]) ? $options_selected_default[$v_item['key']]['enabled'] : false,
            ],
            'weight' => [
              '#type' => 'weight',
              '#title' => t('Weight'),
              '#title_display' => 'invisible',
              '#default_value' => isset($options_selected_default[$v_item['key']]) ? $options_selected_default[$v_item['key']]['weight'] : count($agenda_options_getted['aggregations']['items']) +1,
              '#value' => isset($options_selected_default[$v_item['key']]) ? $options_selected_default[$v_item['key']]['weight'] : $k_item,
              '#delta' => count($agenda_options_getted['aggregations']['items']) + 10,
              '#attributes' => [
                'class' => [
                  'draggable-weight'
                ]
              ],
            ],
          ];
        }
      }
    }

    // Change order of fields from weight
    uasort($form['options_selected_container']['options_selected'], function($a, $b) {return ((is_array($a) && isset($a['#weight'])) ? $a['#weight'] : 0) - ((is_array($b) && isset($b['#weight'])) ? $b['#weight'] : 0);});

    return $form;
  }

  /**
   * Callback ajax to get filter Options from agenda and filter type.
   * @param mixed $form
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   */
  public function ajaxGetFilterOptionsCallback($form, FormStateInterface $form_state){
    // Update field options selected
    return $form['settings']['options_selected_container'];
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state)
  {
    parent::blockSubmit($form, $form_state);
    $values = $form_state->getValues();
    $this->configuration['input_placeholder'] = $values['input_placeholder'];
    $this->configuration['input_aria_label'] = $values['input_aria_label'];
    $this->configuration['filter_type'] = $values['filter_type'];
    $this->configuration['options_page_size'] = $values['options_page_size'];
    $this->configuration['options_limited'] = $values['options_limited'];
    $this->configuration['options_agenda'] = $values['options_agenda'];
    $this->configuration['option_tag'] = $values['option_tag'];

    if(!empty($values['filter_type']) && !empty($values['options_agenda']) && !empty($values['options_selected_container'])){
      $options_selected_submitted = $values['options_selected_container']['options_selected'];

      // Change order of options
      uasort($options_selected_submitted, function($a, $b) {return $a['weight'] - $b['weight'];});

      $options_selected = $this->configuration['options_selected'] ?? [];
      $options_selected[$values['options_agenda']][$values['filter_type']] = $options_selected_submitted ;
      $this->configuration['options_selected'] = $options_selected;
    }
  }

  /**
   * @return int
   *   Cache max age.
   */
  public function getCacheMaxAge()
  {
    return 0;
  }

}
