<?php

namespace Drupal\openagenda\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\openagenda\OpenagendaHelperInterface;
use OpenAgendaSdk\OpenAgendaSdk;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines the OpenAgenda settings form.
 */
class OpenagendaForm extends ConfigFormBase {

  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typed_config_manager,
    protected OpenagendaHelperInterface $helper,
  ) {
    parent::__construct($config_factory, $typed_config_manager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('openagenda.helper'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'openagenda_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildForm($form, $form_state);

    $form['public_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('OpenAgenda public key'),
      '#description' => $this->t('Enter your OpenAgenda account public key.'),
      '#config_target' => 'openagenda.settings:openagenda.public_key',
      '#required' => TRUE,
    ];

    $form['general_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('General settings'),
    ];

    $form['general_settings']['oac_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Load navigation context in links pointing to detail pages'),
      '#description' => $this->t("The navigation context allows the addition of 'Next' and 'Previous' links on the detail pages, allowing the visitor to navigate directly between the results of his search without systematically returning to the list view."),
      '#return_value' => TRUE,
      '#config_target' => 'openagenda.settings:openagenda.oac_enabled',
    ];

    $form['general_settings']['tracking_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('I authorize OpenAgenda to keep track of the use of the module for statistical purposes on the following information: the cms used, the domain of the site hosting the extension.'),
      '#return_value' => TRUE,
      '#config_target' => 'openagenda.settings:openagenda.tracking_enabled',
    ];

    $form['general_settings']['manual_submit'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Disable automatic event search when filters change'),
      '#return_value' => TRUE,
      '#config_target' => 'openagenda.settings:openagenda.manual_submit',
    ];

    $form['general_settings']['daterange_simple'] = [
      '#type' => 'checkbox',
      '#title' => $this->t("Use the pair of simple 'date' HTML fields to filter by dates"),
      '#return_value' => TRUE,
      '#config_target' => 'openagenda.settings:openagenda.daterange_simple',
    ];

    $form['default_openagenda_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Default OpenAgenda content settings'),
    ];

    $form['default_openagenda_settings']['events_per_page'] = [
      '#type' => 'number',
      '#title' => $this->t('Events per page'),
      '#description' => $this->t('Number of events displayed per page. Enter 0 to show all events.'),
      '#min' => 0,
      '#max' => 300,
      '#size' => 3,
      '#config_target' => 'openagenda.settings:openagenda.events_per_page',
    ];

    $language_options = ['default' => $this->t("Use site's language")] + $this->helper->getAvailableLanguages();

    $form['default_openagenda_settings']['default_language'] = [
      '#type' => 'select',
      '#title' => $this->t('Default language'),
      '#description' => $this->t('The default language to use for OpenAgendas and events.'),
      '#options' => $language_options,
      '#config_target' => 'openagenda.settings:openagenda.default_language',
    ];

    $form['default_openagenda_settings']['include_embedded'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include embedded content'),
      '#description' => $this->t('Include embedded HTML content in event descriptions.'),
      '#return_value' => TRUE,
      '#config_target' => 'openagenda.settings:openagenda.include_embedded',
    ];

    $form['default_openagenda_settings']['current'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Only current and upcoming events'),
      '#description' => $this->t('Display only current and upcoming events. If relative date filter block is configured to be displayed on agenda page, it will be disabled on every openagenda node page.'),
      '#return_value' => TRUE,
      '#config_target' => 'openagenda.settings:openagenda.current',
    ];

    $form['default_openagenda_display'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Default OpenAgenda display settings'),
    ];

    $form['default_openagenda_display']['default_style'] = [
      '#type' => 'select',
      '#title' => $this->t('OpenAgenda default style'),
      '#options' => [
        'agenda' => $this->t('OpenAgenda'),
        "default" => $this->t('None'),
      ],
      '#config_target' => 'openagenda.settings:openagenda.default_style',
    ];

    $form['default_openagenda_display']['default_columns'] = [
      '#type' => 'select',
      '#title' => $this->t('Events per column'),
      '#options' => [
        1 => 1,
        2 => 2,
        3 => 3,
        4 => 4,
      ],
      '#config_target' => 'openagenda.settings:openagenda.default_columns',
    ];

    $form['default_map_filter_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Default map filter settings'),
    ];

    $form['default_map_filter_settings']['default_map_filter_tiles_uri'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Default map tiles URL'),
      '#description' => $this->t('Default URL of the map tiles used for the filter.'),
      '#config_target' => 'openagenda.settings:openagenda.default_map_filter_tiles_uri',
    ];

    $form['default_map_filter_settings']['default_map_filter_tiles_attribution'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Default map tiles attribution'),
      '#description' => $this->t('Default map tiles attribution to display on filter.'),
      '#config_target' => 'openagenda.settings:openagenda.default_map_filter_tiles_attribution',
    ];

    $form['default_map_filter_settings']['default_map_filter_label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Default map filter label'),
      '#description' => $this->t('Text to use as the map filter input field label.'),
      '#config_target' => 'openagenda.settings:openagenda.default_map_filter_label',
    ];

    $form['default_search_filter_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Default search filter settings'),
    ];

    $form['default_search_filter_settings']['default_search_filter_placeholder'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Default search filter placeholder'),
      '#description' => $this->t('Text to use as the search filter input field placeholder.'),
      '#config_target' => 'openagenda.settings:openagenda.default_search_filter_placeholder',
    ];
    $form['default_search_filter_settings']['default_choice_filter_placeholder'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Default choice filter placeholder'),
      '#description' => $this->t('Text to use as the choice filter input field placeholder.'),
      '#config_target' => 'openagenda.settings:openagenda.default_choice_filter_placeholder',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $public_key = $form_state->getValue('public_key');
    if (!empty($public_key)) {
      // Prepare SDK instance with the public key.
      $sdk = new OpenAgendaSdk($form_state->getValue('public_key'));
      // Get agendas to test the connection.
      $my_agendas = \json_decode($sdk->getMyAgendas(), TRUE);
      // If has error.
      if (array_key_exists('error', $my_agendas)) {
        $form_state->setErrorByName('public_key', $this->t('The public key is not valid. Please check your OpenAgenda public key.'));
      }
      elseif (empty($my_agendas)) {
        $form_state->setErrorByName('public_key', $this->t('No agendas found for the provided public key. Please check your OpenAgenda public key.'));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'openagenda.settings',
    ];
  }

}
